<?php
namespace Dalten\WebBundle\Processor\PropertyDisplayProvider;

use Dalten\Date\Date;
use Dalten\WebBundle\Entity\EntityWithLegacyData;
use Dalten\WebBundle\Entity\Variable;
use Doctrine\ORM\EntityRepository;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Processor na získávání textových reprezentací číselníkových položek.
 */
class Base
{
	/**
	 * Pole názvů (varNames) okolonek, které obsahují měnu.
	 *
	 * @var array
	 */
	private $_currencyFields = array('nemovitost_cena_prodej', 'nemovitost_cena_pronajem');

	/**
	 * Přípona k názvu vlastnosti, označujícím plochu.
	 */
	const AREA_PROPERTY_SUFFIX = '_plocha';

	/**
	 * Repozitář proměnných.
	 *
	 * @var EntityRepository
	 */
	private $_variablesRepository;

	/**
	 * Pomocné pole, obsahující variables, indexované podle jména.
	 *
	 * @var Variable[]
	 */
	private $_variablesIndexedByName = array();

	/**
	 * Translator.
	 *
	 * @var TranslatorInterface
	 */
	protected $_translator;

	/**
	 * Pole číselníků - používá se pro typy variables, které nemají vlastní číselníky (např. ano/ne/?).
	 *
	 * @var array
	 */
	protected $_codebooks;

	/**
	 * Interní (lazy-loaded) instance number-formatteru, klíčované podle jazyka.
	 *
	 * @var \NumberFormatter[]
	 */
	protected $_numberFormatters;

	/**
	 * Interní (lazy-loaded) instance number-formatteru pro měnu, klíčované podle jazyka.
	 *
	 * @var \NumberFormatter[]
	 */
	protected $_currencyNumberFormatters;

	/**
	 * Interní (lazy-loaded) instance intl-date-formatteru, klíčované podle jazyka a typu data/času.
	 *
	 * @var \IntlDateFormatter[]
	 */
	protected $_dateFormatters;

	/**
	 * Defaultní jazyk.
	 *
	 * @var string
	 */
	protected $_defaultLanguage;

	/**
	 * Povolené jazyky.
	 *
	 * Pole, kde klíč je dvouznakový kód a hodnota je suffix lang-* v DB u variables.
	 * Tj.: array('cs' => 'czech');
	 *
	 * @var array
	 */
	protected $_allowedLanguages;

	/**
	 * Nastavuje závislosti a konfiguraci.
	 *
	 * @param EntityRepository       $variablesRepository     Repozitář variables.
	 * @param TranslatorInterface    $translator              Překladač.
	 * @param array                  $codebooks               Pole číselníků.
	 * @param array                  $allowedLanguages        Pole povolených jazyků (viz popis proměnné)
	 * @param string                 $defaultLanguage         Defaultní jazyk (dvouznakový kód).
	 */
	public function __construct(EntityRepository $variablesRepository, TranslatorInterface $translator,
		array $codebooks, array $allowedLanguages, $defaultLanguage)
	{
		$this->_variablesRepository = $variablesRepository;
		$this->_translator = $translator;
		$this->_codebooks = $codebooks;
		$this->_defaultLanguage = $defaultLanguage;
		$this->_allowedLanguages = $allowedLanguages;
	}

	/**
	 * Nahraje data pro daná varNames a kategorii.
	 *
	 * Již nahraná data (ta co jsou v cache) znovu nevyhledává.
	 *
	 * @param array    $varNames Pole variable names z DB.
	 * @param null|int $category Id kategorie.
	 */
	public function loadData(array $varNames = array(), $category = null)
	{
		if ($varNames && $this->_variablesIndexedByName) {
			$varNames = array_diff($varNames, array_keys($this->_variablesIndexedByName));
			if (empty($varNames)) {
				return;
			}
		}

		$queryBuilder = $this->_variablesRepository->createQueryBuilder('v')
			->leftJoin('v.listValues', 'lv')
			->select('v, lv');

		if ($varNames) {
			$queryBuilder->where('v._var_name IN (:var_names)');
			$queryBuilder->setParameter('var_names', $varNames);
		}
		if ($category) {
			$queryBuilder->where('v._id_category = :id_category');
			$queryBuilder->setParameter('id_category', $category);
		}

		$variables = $queryBuilder->getQuery()->getResult();

		foreach ($variables as $variable) {
			$this->_variablesIndexedByName[$variable->variable_name] = $variable;
		}
	}

	/**
	 * Vrátí asociativní pole, kde klíče jsou názvy vlastností nabídky a hodntoy jsou hodnoty těchto vlastností.
	 *
	 * Klíče i hodnoty jsou již přeloženy do požadovaného jazyka a není třeba je překládat znovu.
	 *
	 * @param EntityWithLegacyData $entity            Entita.
	 * @param array                $varNames          Pole názvů vlastností ve formátu var_names.
	 * @param string               $language          Požadovaný jazyk.
	 * @param bool                 $ignoreEmptyValues Mají se vracet jen neprázdné hodnoty?
	 *
	 * @return array Pole s názvy (klíče) a hodnotami (hodnoty) vlastnosti nabídky.
	 */
	public function getLabelAndValuePairs(EntityWithLegacyData $entity, array $varNames, $language, $ignoreEmptyValues)
	{
		$this->loadData($varNames);

		$return = array();

		foreach ($varNames as $propertyName) {
			if ($propertyName == 'kod' && method_exists($entity, 'getLongCode')) {
				$key = $this->_translator->trans('Číslo zakázky');
				$return[$key] = $entity->getLongCode();
				continue;
			}

			if (!isset($this->_variablesIndexedByName[$propertyName])) {
				continue;
			}

			/** @var Variable $propertyModel */
			$propertyModel = $this->_variablesIndexedByName[$propertyName];

			$key = $this->getPropertyLabel($propertyModel, $language);
			$propertyValue = $entity->getLegacyProperty($propertyModel->variable_name);

			if (!$ignoreEmptyValues || !empty($propertyValue)) {
				$value = $this->getPropertyValue($propertyModel, $propertyValue, $language, $entity);

				$return[$key] = $value;
			}

		}

		return $ignoreEmptyValues ? array_filter($return) : $return;
	}

	/**
	 * Vrátí název vlastnosti pro daný jazyk.
	 *
	 * @param Variable $variable Instance proměnné.
	 * @param string   $language Kód jazyka (cs).
	 *
	 * @return string Název vlastnosti nabídky.
	 */
	public function getPropertyLabel(Variable $variable, $language)
	{
		$this->_checkLanguage($language);

		return $variable->{$this->_getVariablePropertyNameForLanguage($language)};
	}

	/**
	 * Vrátí hodnotu vlastnosti nabídky pro daný jazyk, formátovanou pro vypsání uživateli.
	 *
	 * Pokud má variable nepodporovaný typ, bude vrácena $value bez změny.
	 *
	 * $context by měl být ve formátu polí s DB (s ohledem na suffix _plocha).
	 *
	 * @param Variable             $variable Instance proměnné.
	 * @param mixed                $value    Hodnota vlastnosti.
	 * @param string               $language Kód jazyka (cs).
	 * @param EntityWithLegacyData $entity   Kontext. Entita k získávání YESNOAREA vlastností.
	 *
	 * @return string Hodnota vlastnosti nabídky, formátovaná pro vypsání uživateli.
	 *
	 */
	public function getPropertyValue(Variable $variable, $value, $language, EntityWithLegacyData $entity = null)
	{
		$this->_checkLanguage($language);

		$commonCodebookName = null;

		// napřed vyřadíme interní číselníky (checkbox, yesno, yesnomaybe, yesnoquestion, currency)
		if (isset($this->_codebooks[$variable->type])) {
			$translatedValue = isset($this->_codebooks[$variable->type][$value])
				? $this->_codebooks[$variable->type][$value] : null;

			return $translatedValue ?
				$this->_translator->trans($translatedValue, array(), null, $language) : $translatedValue;
		}

		switch ($variable->type) {
			case Variable::TYPE_DATE:
				if (empty($value) || $value == '0000-00-00') {
					return null;
				}
				$dt = $value instanceof Date ? $value : new Date($value);

				return $this->_getDateTimeFormatter($language, \IntlDateFormatter::MEDIUM, \IntlDateFormatter::NONE)
					->format($dt->getDateTime());

			case Variable::TYPE_FLOAT:
				if (in_array($variable->variable_name, $this->_currencyFields)) {
					return str_replace(
						'¤', '', $this->_getCurrencyFormatter($language)->format($value, \NumberFormatter::CURRENCY)
					);
				}
				return $this->_getNumberFormatter($language)->format($value, \NumberFormatter::TYPE_DOUBLE);

			case Variable::TYPE_INT:
				return $this->_getNumberFormatter($language)->format($value, \NumberFormatter::TYPE_INT32);

			case Variable::TYPE_INT_METERS:
				$formattedValue = $this->_getNumberFormatter($language)->format($value, \NumberFormatter::TYPE_INT32);

				return $this->_translator->trans('#cislo# m', array('#cislo#' => $formattedValue), null, $language);

			case Variable::TYPE_INT_SQUARE_METERS:
				$formattedValue = $this->_getNumberFormatter($language)->format($value, \NumberFormatter::TYPE_INT32);

				return $this->_translator->trans('#cislo# m²', array('#cislo#' => $formattedValue), null, $language);

			case Variable::TYPE_YESNOAREA:
				if (!isset($this->_codebooks[Variable::TYPE_YESNO][$value]) || $value == -1) {
					return '';
				}

				$translatedValue = $this->_translator->trans(
					$this->_codebooks[Variable::TYPE_YESNO][$value], array(), null, $language
				);

				$areaPropertyName = $variable->variable_name . self::AREA_PROPERTY_SUFFIX;
				$formattedArea = null;
				$area = $entity->getLegacyProperty($areaPropertyName);
				if ($value > 0 && $area > 0) {
					$formattedArea = $this->_getNumberFormatter($language)->format($area, \NumberFormatter::TYPE_INT32);
					$formattedArea = $this->_translator
						->trans('#cislo# m²', array('#cislo#' => $formattedArea), null, $language);

				}

				return $formattedArea ? sprintf('%s (%s)', $translatedValue, $formattedArea) : $translatedValue;

			case Variable::TYPE_MULTICHECKBOX:
				$labels = array();
				$parsedValue = $value;
				if (is_string($value)) {
					$parsedValue = array();
					foreach (str_split($value) as $index => $isSet) {
						if ($isSet) {
							$parsedValue[] = $index;
						}
					}
				} else {
					$parsedValue = (array) $parsedValue;
				}
				foreach ($parsedValue as $codebookValue) {
					$labels[] = $this->getCodebookLabel($variable, $codebookValue, $language);
				}

				return implode(', ', array_filter($labels));

			case Variable::TYPE_SELECT:
				return $this->getCodebookLabel($variable, $value, $language);

			case Variable::TYPE_MEMO:
				return nl2br($value);
		}

		return $value;
	}

	/**
	 * Zkontroluje, zda je daný jazyk podporován. Pokud ne, vyhodí InvalidArgumentException.
	 *
	 * @param string $language Kód jazyka (cs).
	 */
	protected function _checkLanguage($language)
	{
		if (!isset($this->_allowedLanguages[$language])) {
			throw new \InvalidArgumentException(
				sprintf(
					'Neznámý jazyk "%s". Podporované jazyky: "%s".',
					$language,
					implode('", "', array_keys($this->_allowedLanguages))
				)
			);
		}
	}

	/**
	 * Vrátí label číselníkové hodnoty s danou value pro daný jazyk.
	 *
	 * @param Variable $variable Instance proměnné.
	 * @param mixed    $value    Hodnota vlastnosti (proměnné).
	 * @param string   $language Kód jazyka (cs).
	 *
	 * @return string|null Label dané hodnoty nebo null.
	 */
	public function getCodebookLabel(Variable $variable, $value, $language)
	{
		$this->_checkLanguage($language);
		$listValue = $variable->getListValueByValue($value);
		$key = $this->_getVariablePropertyNameForLanguage($language);

		return isset($listValue->$key) ? $listValue->$key : null;
	}

	/**
	 * Vrátí název vlastnosti listValue pro daný jazyk.
	 *
	 * Data pro převod jsou brána z $this->_allowedLanguages.
	 * Před voáním se předpokládá ověření existence jazyka.
	 *
	 * Např. ('cs') => 'lang_czech'.
	 *
	 * @param string $language Kód jazyka (cs).
	 *
	 * @return string String lang_*.
	 */
	protected function _getVariablePropertyNameForLanguage($language)
	{
		return 'label_' . $this->_allowedLanguages[$language];
	}

	/**
	 * Vrátí (lazy-loaded) instanci NumberFormatteru pro daný jazyk.
	 *
	 * @param string $lang Kód jazyka.
	 *
	 * @return \NumberFormatter
	 */
	protected function _getNumberFormatter($lang)
	{
		if (!isset($this->_numberFormatters[$lang])) {
			$this->_numberFormatters[$lang] = new \NumberFormatter($lang, \NumberFormatter::DECIMAL);
		}

		return $this->_numberFormatters[$lang];
	}

	/**
	 * Vrátí (lazy-loaded) instanci NumberFormatteru pro daný jazyk.
	 *
	 * @param string $lang Kód jazyka.
	 *
	 * @return \NumberFormatter
	 */
	protected function _getCurrencyFormatter($lang)
	{
		if (!isset($this->_currencyNumberFormatters[$lang])) {
			$this->_currencyNumberFormatters[$lang] = new \NumberFormatter($lang, \NumberFormatter::CURRENCY);
		}

		return $this->_currencyNumberFormatters[$lang];
	}

	/**
	 * Vrátí (lazy-loaded) instanci IntlDateFormatteru pro daný jazyk a typ data.
	 *
	 * @param string $lang Kód jazyka.
	 *
	 * @return \IntlDateFormatter
	 */
	protected function _getDateTimeFormatter($lang, $dateFormat, $timeFormat)
	{
		$key = sprintf('%s-%s-%s', $lang, $dateFormat, $timeFormat);

		if (!isset($this->_dateFormatters[$key])) {
			$this->_dateFormatters[$key] = new \IntlDateFormatter($lang, $dateFormat, $timeFormat);
		}

		return $this->_dateFormatters[$key];
	}
}
