<?php
namespace Dalten\WebBundle\Processor\PropertyDisplayProvider;

use Dalten\WebBundle\Entity\Listing;
use Doctrine\ORM\EntityRepository;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Processor na získávání textových reprezentací číselníkových položek.
 */
class ListingDetail extends Base
{
	/**
	 * Konfigurace názvů zobrazovaných proměnných (očekáváné klíče: start, end, type_<číslo>, type_default).
	 * Pod klíči jsou pole položek ve formátu názvů z DB.
	 *
	 * Položky budou zobrazeny v následující pořadí:
	 * 1) start
	 * 2) type_<číslo> (pokud existuje - jinak type_default)
	 * 3) end
	 *
	 * Položky v rámci skupiny budou zobrazeny v pořadí z configu.
	 *
	 * @var array
	 */
	private $_displayPropertiesConfig;

	/**
	 * Mapa typu nabídky na defaultní plochu.
	 *
	 * @var array
	 */
	private $_listingTypeToAreaMap;

	/**
	 * Nastavuje závislosti a konfiguraci.
	 *
	 * @param EntityRepository    $variablesRepository     Repozitář variables.
	 * @param TranslatorInterface $translator              Překladač.
	 * @param array               $codebooks               Pole číselníků.
	 * @param array               $listingTypeToAreaMap    Mapa typu nemovitosti na název defaultní plochy.
	 * @param array               $displayPropertiesConfig Nastavení zobrazovaných properties (viz popis proměnné).
	 * @param array               $allowedLanguages        Pole povolených jazyků (viz popis proměnné)
	 * @param string              $defaultLanguage         Defaultní jazyk (dvouznakový kód).
	 */
	public function __construct(EntityRepository $variablesRepository, TranslatorInterface $translator,
		array $codebooks, array $listingTypeToAreaMap, array $displayPropertiesConfig, array $allowedLanguages,
		$defaultLanguage)
	{
		$this->_displayPropertiesConfig = $displayPropertiesConfig;
		$this->_listingTypeToAreaMap = $listingTypeToAreaMap;
		parent::__construct($variablesRepository, $translator, $codebooks, $allowedLanguages, $defaultLanguage);
	}

	/**
	 * Vrátí pole názvů vlastností, která mají být použita pro danou nabídku.
	 *
	 * Vracené položky jsou určeny pomocí $this->_displayPropertiesConfig a budou zobrazeny v následující pořadí:
	 * 1) start
	 * 2) type_<číslo> (pokud existuje - jinak type_default)
	 * 3) end
	 *
	 * Položky v rámci skupiny budou zobrazeny v pořadí z $this->_displayPropertiesConfig.
	 *
	 * @param Listing $listing  Model nabídky.
	 * @param string  $language Kód požadovaného jazyka (dvouznakový).
	 *
	 * @return array Pole názvů vlastností pro danou nabídku.
	 */
	public function getPropertyNamesForListing(Listing $listing, $language)
	{
		$this->_checkLanguage($language);

		$typeKey = 'type_' . $listing->listing_type;
		if (!isset($this->_displayPropertiesConfig[$typeKey])) {
			$typeKey = 'type_default';
		}

		$properties = array_merge(
			$this->_displayPropertiesConfig['start'],
			$this->_displayPropertiesConfig[$typeKey],
			$this->_displayPropertiesConfig['end']
		);

		$langSuffix = $language == $this->_defaultLanguage ? '' : '_' . $language;

		foreach ($properties as $key => $property) {
			$properties[$key] = preg_replace('/#lang#$/', $langSuffix, $property);
			if ($property == '__plocha__') {
				if ($listing->listing_type
					&& isset($this->_listingTypeToAreaMap[$listing->listing_type])
				) {
					$properties[$key] = $this->_listingTypeToAreaMap[$listing->listing_type];
				} else {
					unset($properties[$key]);
				}
			}
		}

		return array_unique($properties);
	}
}
