<?php
namespace Dalten\WebBundle\Processor\SeoUrl;

use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Processor\SeoUrl\Exception\UndefinedLanguageException;
use Dalten\WebBundle\SeoRoute\FragmentCollection;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Routing\RouterInterface;

/**
 * Created by PhpStorm.
 * User: karel
 * Date: 19.04.2016
 * Time: 13:55
 */
class RedirectResponseGenerator
{
	/**
	 * Pole ignorovaných parametrů z query.
	 *
	 * Za ignorované jsou považovány ty, které neovlivňují filtr - tj. třeba stránkování, řazení, atp.
	 *
	 * @var array
	 */
	private $_ignoredQueryParams = [];

	/**
	 * Pole matcherů na seo url.
	 *
	 * Klíčem je APLHA2 jazyka a hodnotou je instance matcheru.
	 *
	 * @var FragmentCollection[]
	 */
	private $_matchers = [];

	/**
	 * @var RouterInterface
	 */
	private $_router;

	/**
	 * RedirectResponseGenerator constructor.
	 *
	 * @param RouterInterface                                 $router
	 * @param \Dalten\WebBundle\SeoRoute\FragmentCollection[] $matchers
	 * @param array                                           $ignoredQueryParams
	 */
	public function __construct(RouterInterface $router, array $matchers, array $ignoredQueryParams)
	{
		$this->_ignoredQueryParams = $ignoredQueryParams;
		$this->_matchers = $matchers;
		$this->_router = $router;
	}

	/**
	 * Vrátí redirect response na SEO url - pokud je možné takovou URL vytvořit.
	 *
	 * Pokud jsou v query i jiné než ignorované parametry (viz konstruktor), tak se o redirect ani nepokusí.
	 *
	 * @param ListingFilter $filter      Instance filtru.
	 * @param array         $queryParams Pole parametrů z query.
	 * @param string        $locale      Kód locale (APLHA2).
	 *
	 * @return null|RedirectResponse Redirect na SEO URL a nebo null.
	 *
	 * @throws UndefinedLanguageException Pokud není nastaven matcher pro daný jazyk.
	 */
	public function getSeoUrlRedirectResponse(ListingFilter $filter, array $queryParams, $locale)
	{
		$excludedParams = array();
		$cleanedQuery = $queryParams;
		foreach ($this->_ignoredQueryParams as $ignoredFieldName) {
			if (array_key_exists($ignoredFieldName, $cleanedQuery)) {
				$excludedParams[$ignoredFieldName] = $cleanedQuery[$ignoredFieldName];
				unset($cleanedQuery[$ignoredFieldName]);
			}
		}

		$attemptRedirectToSeoUrl = !empty($cleanedQuery);

		if ($attemptRedirectToSeoUrl) {
			if (!isset($this->_matchers[$locale])) {
				throw new UndefinedLanguageException(
					sprintf(
						'Není nastaven SeoUrlMatcher pro locale "s". [Podporované locales: %s]',
						$locale,
						implode(
							', ', array_map(function($value) {return '"' . $value . '"';}, array_keys($this->_matchers))
						)
					)
				);
			}

			$routeConfig = $this->_matchers[$locale]->getRouteConfig($filter);

			if ($routeConfig) {
				$routeConfig['parameters'] = array_merge($routeConfig['parameters'], $excludedParams);
				$url = $this->_router->generate($routeConfig['name'], $routeConfig['parameters']);

				return new RedirectResponse($url, 301);
			}
		}

		return null;
	}
}
