<?php
namespace Dalten\WebBundle\Processor\TitleGenerator\ListingList;

use Dalten\WebBundle\Filter\ListingFilter;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Generátor na titulky nemovitostí, založené na podtypu (nikoli na typu).
 *
 * Slouží vždy pouze pro jeden typ nemovitosti a pro něj vytváří titulky na základě podtypů.
 */
class ListingSubtype extends Base implements GeneratorInterface
{
	/**
	 * Text, přidaný před typy nemovitostí.
	 *
	 * @var int
	 */
	private $_requiredListingTypeValue;

	/**
	 * Název vlastnosti filtru, ve které je podtyp.
	 *
	 * @var string
	 */
	private $_subtypePropertyName;

	/**
	 * Nastavuje překladač a různé číselníky.
	 *
	 * $codebooks je asociativní pole ve formátu array([název číselníku] => array([hodnota] => [label], ...), ...).
	 * $specialTitles je pole ve formátu array(array('requirements' => array(...jako filtr...), 'title_parts' =>
	 *      array('advert_type' => 'prodej').
	 *
	 * V $codebooks jsou očekávány klíče listing_subtype_default, listing_subtype_lease a listing_subtype_sale.
	 *
	 * @param TranslatorInterface $translator               Instance překladače.
	 * @param GeneratorInterface  $addressTitleGenerator    Generátor části titulku na adresu.
	 * @param array               $codebooks                Číselníky (popis viz popis metody).
	 * @param array               $specialTitles            Pole konfigurace speciálních titulků (viz výše).
	 * @param null                $requiredListingTypeValue Požadovaný typ nemovitosti pro generátor.
	 * @param null                $subtypePropertyName      Název vlastnosti filtru s podtypem.
	 */
	public function __construct(TranslatorInterface $translator, GeneratorInterface $addressTitleGenerator,
		array $codebooks, $specialTitles = array(), $requiredListingTypeValue = null, $subtypePropertyName = null)
	{
		$this->_requiredListingTypeValue = $requiredListingTypeValue;
		$this->_subtypePropertyName = $subtypePropertyName;
		parent::__construct($translator, $addressTitleGenerator, $codebooks, $specialTitles);
	}

	/**
	 * Vytvoří z předaného filtru titulek pro výpis.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return string Vytvořený titulek pro filtr.
	 */
	public function createTitle(ListingFilter $filter)
	{
		// řešíme pouze pokud je vybraný pouze určený typ nemovitosti a máme nějaký podtyp
		if (empty($this->_requiredListingTypeValue)
			|| $filter->listing_type !== array($this->_requiredListingTypeValue)
			|| empty($filter->{$this->_subtypePropertyName})) {
			return '';
		}

		$titleParts = $this->_findSpecialTitleForFilter($filter);

		if (!isset($titleParts['advert_type'])) {
			$titleParts['advert_type'] = $this->_getAdvertTypeTitlePart($filter);
		}

		if (!isset($titleParts['listing_type'])) {
			$titleParts['listing_type'] = $this->_getListingTypePart($filter);
		}

		if (!isset($titleParts['address'])) {
			$titleParts['address'] = $this->_getAddressPart($filter);
		}

		$title = implode(' ', array_filter($titleParts));

		return mb_strtoupper(mb_substr($title, 0, 1)) . mb_substr($title, 1);
	}

	/**
	 * Vrátí název typů (resp. v tomto případě podtypů) nabídky.
	 *
	 * @param \Dalten\WebBundle\Filter\ListingFilter $listingFilter Vyplněný filtr nabídek.
	 *
	 * @return string Název typů (resp. v tomto případě podtypů) nabídky.
	 */
	protected function _getListingTypePart(ListingFilter $listingFilter)
	{
		$subtypeValues = $listingFilter->{$this->_subtypePropertyName};

		return implode(
			', ', array_filter($this->_getTranslatedSubtypeNames($subtypeValues, $listingFilter->advert_type))
		);
	}

	/**
	 * Vrátí přeložený a správně vyskloňovaný název podtypů nemovitosti.
	 *
	 * @param array $listingSubtypes     Podtypy nemovitosti.
	 * @param array $selectedAdvertTypes Pole vybraných typů zakázky (prodej/pronájem).
	 *
	 * @return array Pole přeložených a správně vyskloňovaných názvů podtypu nemovitosti nebo prázdné pole.
	 */
	protected function _getTranslatedSubtypeNames(array $listingSubtypes, array $selectedAdvertTypes)
	{
		$codebookNameSuffix = 'default';
		if ($selectedAdvertTypes == array(1)) {
			$codebookNameSuffix = 'sale';
		} elseif ($selectedAdvertTypes == array(2)) {
			$codebookNameSuffix = 'lease';
		}

		return $this->_getTranslatedCodebookValues($listingSubtypes, 'listing_subtype_' . $codebookNameSuffix);
	}
}
