<?php
namespace Dalten\WebBundle\Tests\Processor\ListingListData;

use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Processor\ListingListData\Builder;
use Mockery as m;
use Symfony\Component\HttpFoundation\Request;

/**
 * Created by PhpStorm.
 * User: Karel
 * Date: 24. 9. 2014
 * Time: 14:46
 */

class BuilderTest extends \PHPUnit_Framework_TestCase
{
	private $_pageVariableName = 'str';

	/**
	 * @dataProvider booleanMethodNamesProvider
	 */
	public function testBooleanParamsAreCorrectlyPassedIntoCreatedDataObject($builderMethodName, $dataMethodName)
	{
		$data = $this->_createEmptyBuilder()->getData();
		$this->assertFalse($data->$dataMethodName(), 'Default by měl být false');

		$data = $this->_createEmptyBuilder()->$builderMethodName(true)->getData();
		$this->assertTrue($data->$dataMethodName());

		$data = $this->_createEmptyBuilder()->$builderMethodName(false)->getData();
		$this->assertFalse($data->$dataMethodName());
	}

	public function testListingFilterIsCorrectlyPassedIntoCreatedDataObject()
	{
		$uriParamsConverter = m::mock('Dalten\WebBundle\Processor\UriParamsToQueryConverter');
		$filterConverter = m::mock('Dalten\WebBundle\Filter\Converter\ListingListInterface');
		$filter = new ListingFilter(array());

		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$builder->setFilter($filter);
		$data = $builder->getData();
		$this->assertSame($filter, $data->getFilter());
	}

	public function testRequestInformationIsCorrectlyPassedIntoCreatedDataObject()
	{
		$uriParamsConverter = m::mock('Dalten\WebBundle\Processor\UriParamsToQueryConverter');
		$filterConverter = m::mock('Dalten\WebBundle\Filter\Converter\ListingListInterface');
		$favoriteListingIds = array(5, 1);
		$queryParams = array('sale' => 2, 'order_by_price' => 0);
		$requestParams = array('has_open_day' => true);
		$conversionResult = array('bagr' => true);
		$filterConversionInput = array_replace_recursive($queryParams, $conversionResult);

		$filter = new ListingFilter(array());

		$uriParamsConverter->shouldReceive('convertParamsToQuery')->once()->with($requestParams)
			->andReturn($conversionResult);
		$filterConverter->shouldReceive('createFilterFromArray')->once()->with($filterConversionInput)
			->andReturn($filter);

		$request = new Request(
			$queryParams,
			array(),
			$requestParams
		);

		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, $favoriteListingIds)->getData();

		$this->assertSame($favoriteListingIds, $data->getFavoriteListingIds());
		$this->assertSame($filter, $data->getFilter());
		$this->assertTrue($data->isOrderedByPrice());
		$this->assertFalse($data->isOrderedByPublishedDate());
		$this->assertTrue($data->isOpenDay());
		$this->assertFalse($data->isDiscounted());
		$this->assertFalse($data->isFavorite());
		$this->assertFalse($data->isForeign());
		$this->assertFalse($data->isInAscendingOrder());
	}

	public function testOpenDaysAreSetAsTrueIfPresentInFilterAndNotInRouteAttributes()
	{
		$uriParamsConverter = m::mock('Dalten\WebBundle\Processor\UriParamsToQueryConverter');
		$filterConverter = m::mock('Dalten\WebBundle\Filter\Converter\ListingListInterface');

		$filter = new ListingFilter(array('has_open_day' => true));

		$uriParamsConverter->shouldReceive('convertParamsToQuery')->never();
		$filterConverter->shouldReceive('createFilterFromArray')->once()->andReturn($filter);

		$request = new Request(array('has_open_day' => 'on'));

		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, array())->getData();

		$this->assertTrue($data->isOpenDay());
	}


	public function testOrderingInformationIsCorrectlyPassedFromRequestIntoCreatedDataObject()
	{
		$uriParamsConverter = m::mock('Dalten\WebBundle\Processor\UriParamsToQueryConverter');
		$filterConverter = m::mock('Dalten\WebBundle\Filter\Converter\ListingListInterface');
		$attrs = array('test' => 'jop');

		$uriParamsConverter->shouldReceive('convertParamsToQuery')->with($attrs)->once()->andReturn(array());
		$filterConverter->shouldReceive('createFilterFromArray')->times(2)->andReturn(new ListingFilter(array()));

		$request = new Request(array('order_by_price' => 1));

		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, array())->getData();

		$this->assertTrue($data->isOrderedByPrice());
		$this->assertFalse($data->isOrderedByPublishedDate());
		$this->assertTrue($data->isInAscendingOrder());

		$request = new Request(array('order_by_published_date' => 0), array(), $attrs);

		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, array())->getData();

		$this->assertFalse($data->isOrderedByPrice());
		$this->assertTrue($data->isOrderedByPublishedDate());
		$this->assertFalse($data->isInAscendingOrder());
	}

	public function testListTypeIsCorrectlyPassedFromRequestIntoCreatedDataObject()
	{
		$uriParamsConverter = m::mock('Dalten\WebBundle\Processor\UriParamsToQueryConverter');
		$filterConverter = m::mock('Dalten\WebBundle\Filter\Converter\ListingListInterface');

		$uriParamsConverter->shouldReceive('convertParamsToQuery')->times(3)->andReturn(array());
		$filterConverter->shouldReceive('createFilterFromArray')->times(3)->andReturn(new ListingFilter(array()));

		$request = new Request(array(), array(), array('_foreignListings' => true));
		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, array())->getData();

		$this->assertTrue($data->isForeign());
		$this->assertFalse($data->isDiscounted());
		$this->assertFalse($data->isFavorite());
		$this->assertFalse($data->isOpenDay());


		$request = new Request(array(), array(), array('_favourites' => true));
		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, array())->getData();

		$this->assertTrue($data->isFavorite());
		$this->assertFalse($data->isDiscounted());
		$this->assertFalse($data->isForeign());
		$this->assertFalse($data->isOpenDay());


		$request = new Request(array(), array(), array('discount_duration' => true));
		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$data = $builder->buildFromRequest($request, array())->getData();

		$this->assertTrue($data->isDiscounted());
		$this->assertFalse($data->isForeign());
		$this->assertFalse($data->isFavorite());
		$this->assertFalse($data->isOpenDay());
	}

	public function testFavoriteListingIdsAreCorrectlyPassedIntoCreatedDataObject()
	{
		$builder = $this->_createEmptyBuilder();
		$data = $builder->getData();
		$this->assertSame(array(), $data->getFavoriteListingIds(), 'default pro $favoriteListingIds je prázdné pole');
		$favoriteListingIds = array(1, 7, 5);

		$data = $this->_createEmptyBuilder()->setFavoriteListingIds($favoriteListingIds)->getData();
		$this->assertSame($favoriteListingIds, $data->getFavoriteListingIds());
	}

	public function testPageNumberIsCorrectlyPassedIntoCreatedDataObject()
	{
		$data = $this->_createEmptyBuilder()->getData();
		$this->assertSame(1, $data->getCurrentPageNumber(), 'Default pro $currentPageNumber má být 1');
		$pageNumber = 2;
		$data = $this->_createEmptyBuilder()->setPageNumber($pageNumber)->getData();
		$this->assertSame($pageNumber, $data->getCurrentPageNumber());
	}

	public function testOrderingParamsAreCorrectlyPassedIntoCreatedDataObject()
	{
		$data = $this->_createEmptyBuilder()->getData();
		$this->assertTrue($data->isOrderedByPublishedDate(), 'Default pro řazení je podle data vytvoření');
		$this->assertFalse($data->isOrderedByPrice(), 'Default pro řazení je podle data vytvoření');
		$this->assertFalse($data->isInAscendingOrder(), 'Default pro směr řazení je sestupně');

		$data = $this->_createEmptyBuilder()->orderByPrice()->getData();
		$this->assertFalse($data->isOrderedByPublishedDate());
		$this->assertTrue($data->isOrderedByPrice());
		$this->assertTrue($data->isInAscendingOrder());

		$data = $this->_createEmptyBuilder()->orderByPublishedDate(false)->getData();
		$this->assertTrue($data->isOrderedByPublishedDate());
		$this->assertFalse($data->isOrderedByPrice());
		$this->assertFalse($data->isInAscendingOrder());
	}


	public function booleanMethodNamesProvider()
	{
		return array(
			'favorite' => array('setIsFavorite', 'isFavorite'),
			'discounted' => array('setIsDiscounted', 'isDiscounted'),
			'foreign' => array('setIsForeign', 'isForeign'),
			'openDay' => array('setIsOpenDay', 'isOpenDay'),
		);
	}

	/**	 *
	 * @return Builder
	 */
	private function _createEmptyBuilder()
	{
		$uriParamsConverter = m::mock('Dalten\WebBundle\Processor\UriParamsToQueryConverter');
		$filterConverter = m::mock('Dalten\WebBundle\Filter\Converter\ListingListInterface');
		$filter = new ListingFilter(array());

		$builder = new Builder($uriParamsConverter, $filterConverter, $this->_pageVariableName);
		$builder->setFilter($filter);

		return $builder;
	}
}
