<?php

namespace Dalten\Date;

/**
 * Pracuje s časem pomocí objektu DateTime.
 *
 * Veškerá práce s čase se provádí pomocí vnitřního objektu DateTime.
 * Pro výpis se používá Zend_Date a to z důvodů lokalizace.
 *
 * @category Dalten
 * @package  Date
 */
class Date
{
	/**
	 * 24h reprezentace času včetně sekund.
	 */
	const TIME = 'H:i:s';

	/**
	 * 24h reprezentace času bez sekund.
	 */
	const TIME_SHORT = 'H:i';

	/**
	 * Vrátí formát pro vypsání data a času.
	 */
	const DATE_TIME = 'Y-m-d H:i';

	/**
	 * Vrátí formát pro vypsání data.
	 */
	const DATE = 'Y-m-d';

	/**
	 * Instance casoveho objektu.
	 *
	 * @var \DateTime
	 */
	private $_datetime = null;

	/**
	 * Vytvoří vnitřní instanci DateTime a nastaví locale.
	 *
	 * @param string        $time     Textová reprezentace času která je kompatibilní s metodou strtotime().
	 * @param \DateTimeZone $timezone Instance objektu určující časovou zónu.
	 *                                Pokud se nevloží žádná použije se systémová.
	 */
	public function __construct($time = 'now', \DateTimeZone $timezone = null)
	{
		if ($time instanceof \DateTime) {
			$this->_datetime = $time; // DateTime už máme, není důvodho tvořit znovu
		} else {
			// Bohužel konstruktor nepobere jako druhý parametr null a
			// musí se vytvářet tato obezlička. Fuj fujj.
			if ($timezone instanceof \DateTimeZone) {
				$this->_datetime = new \DateTime($time, $timezone);
			} else {
				$this->_datetime = new \DateTime($time);
			}
		}
	}

	/**
	 * Přidá k vnitřnímu času interval.
	 *
	 * @param \DateInterval|string $interval Instance časového intervalu nebo interval ve stringu (1 month).
	 *
	 * @uses _getIntervalObject()
	 *
	 * @return Date
	 */
	public function add($interval)
	{
		$this->_datetime->add($this->_getIntervalObject($interval));
		return $this;
	}

	/**
	 * Přičte k vnitřnímu časovému objektu čas ve formátu HH::MM::SS.
	 *
	 * @param string $time Časový údaj ve formátu HH:MM:SS
	 *
	 * @return Date
	 */
	public function addTime($time)
	{
		$timeResult = \date_parse_from_format('H:i:sP', $time);
		$strInterval = \sprintf('PT%dH%dM%dS', $timeResult['hour'], $timeResult['minute'], $timeResult['second']);
		$this->add(new \DateInterval($strInterval));
		return $this;
	}

	/**
	 * Odečte od vnitřního času interval.
	 *
	 * @param \DateInterval|string $interval Instance časového intervalu nebo interval ve stringu (1 month).
	 *
	 * @uses _getIntervalObject()
	 *
	 * @return Date
	 */
	public function sub($interval)
	{
		$this->_datetime->sub($this->_getIntervalObject($interval));
		return $this;
	}

	/**
	 * Nastaví čas.
	 *
	 * Pro nastavení času použije volání metody setDate nad objektem \DateTime.
	 *
	 * @param integer $year  Rok.
	 * @param integer $month Měsíc.
	 * @param integer $day   Den.
	 *
	 * @uses \DateTime::setDate()
	 *
	 * @return Date
	 */
	public function set($year, $month, $day)
	{
		$this->_datetime->setDate($year, $month, $day);
		return $this;
	}

	/**
	 * Porovná vložený čas s vlastní instancí.
	 *
	 * Pokud je objekt ($this) později (větší) než vložený čas ($date) vrátí 1.
	 * Pokud je objekt ($this) shodný s vloženým časem ($date) vrátí 0.
	 * Pokud je objekt ($this) dříve (menší) než vložený čas ($date) vrátí -1.
	 *
	 * $this > $date => 1
	 * $this == $date => 0
	 * $this < $date => -1
	 *
	 * @param mixed   $date     Čas, musí být instancí \Dalten\Date\Date|\DateTime a nebo
	 *                          kompatibilní s funkcí strtotime().
	 * @param boolean $absolute Více v DateTime::diff().
	 *
	 * @return integer Vrací číslici reprezentující pozici v čase.
	 */
	public function compare($date, $absolute = false)
	{
		if ($date === null) {
			return 1;
		}
		$datetime = $this->_createDateTimeObject($date);
		$interval = $datetime->diff($this->getDateTime(), $absolute);
		if ($interval->invert === 1) {
			return -1;
		}

		return $interval->format('%y-%m-%d %h-%i-%s') === '0-0-0 0-0-0' ? 0 : 1;
	}

	/**
	 * Pokud je objekt ($this) větší než vložený čas ($date) vrátí kladnou booleanovskou hodnotu.
	 *
	 * $this > $date => true
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compare()
	 */
	public function isLater($date)
	{
		return $this->compare($date) === 1 ? true : false;
	}

	/**
	 * Pokud je objekt ($this) menší než vložený čas ($date) vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::comapre()
	 */
	public function isLaterOrEqual($date)
	{
		return $this->compare($date) > -1 ? true : false;
	}

	/**
	 * Pokud je objekt menší než vložený čas vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compare()
	 */
	public function isEarlier($date)
	{
		return $this->compare($date) === -1 ? true : false;
	}

	/**
	 * Pokud je objekt menší nebo shodý než vložený čas vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compare()
	 */
	public function isEarlierOrEqual($date)
	{
		return $this->compare($date) < 1 ? true : false;
	}

	/**
	 * Porovná vložený datum s vlastní instancí.
	 *
	 * Pokud je objekty větší než vložený datum vrátí 1.
	 * Pokud je objekt shodný s vloženým datem vrátí 0.
	 * Pokud je objekt shodný menší než vložený datum vrátí -1.
	 *
	 * @param mixed   $date     Datum, musí být instancí \Dalten\Date\Date|\DateTime a nebo
	 *                          kompatibilní s funkcí strtotime().
	 * @param boolean $absolute Více v DateTime::diff().
	 *
	 * @return integer Vrací číslici reprezentující pozici v datu.
	 */
	public function compareDate($date, $absolute = false)
	{
		if ($date === null) {
			return 1;
		}
		$datetime = $this->_createDateTimeObject($date);
		$interval = $datetime->diff($this->getDateTime(), $absolute);
		if ($interval->invert === 1) {
			return -1;
		}

		return $interval->format('%y-%m-%d') === '0-0-0' ? 0 : 1;
	}

	/**
	 * Pokud je objekt větší než vložený datum vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compareDate()
	 */
	public function isDateLater($date)
	{
		return $this->compareDate($date) === 1 ? true : false;
	}

	/**
	 * Pokud je objekt větší nebo rovný než vložený datum vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compareDate()
	 */
	public function isDateLaterOrEqual($date)
	{
		return $this->compareDate($date) > -1 ? true : false;
	}

	/**
	 * Pokud je objekt menší než vložený datum vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compareDate()
	 */
	public function isDateEarlier($date)
	{
		return $this->compareDate($date) === -1 ? true : false;
	}

	/**
	 * Pokud je objekt menší nebo rovný než vložený datum vrátí kladnou booleanovskou hodnotu.
	 *
	 * @param mixed $date Datum musí být kompatibilní s metodou _createDateTimeObject()
	 *
	 * @return boolean
	 *
	 * @uses self::_createDateTimeObject()
	 * @uses self::compareDate()
	 */
	public function isDateEarlierOrEqual($date)
	{
		return $this->compareDate($date) < 1 ? true : false;
	}

	/**
	 * Vrátí vnitřní datum v požadovaném formátu.
	 *
	 * Jedná se o alias k metodě DateTime::format()
	 *
	 * @param string $dateFormat Formát data.
	 *
	 * @return string Naformátované datum.
	 *
	 * @uses DateTime::format()
	 */
	public function format($dateFormat)
	{
		return $this->_datetime->format($dateFormat);
	}

	/**
	 * Vrátí časové razítko.
	 *
	 * @return integer
	 */
	public function getTimestamp()
	{
		return $this->_datetime->getTimestamp();
	}

	/**
	 * Vrátí instanci \DateIntervalu.
	 *
	 * Pokud vstupní parametr není instancí DateInterval
	 * pokusí se pomocí metody DateInterval::createFromDateString() vytvořit jeho instanci.
	 *
	 * @param \DateInterval|string $interval Instance časového intervalu nebo interval ve stringu (1 month).
	 *
	 * @return \DateInterval
	 */
	private function _getIntervalObject($interval)
	{
		if ($interval instanceof \DateInterval) {
			return $interval;
		}
		return \DateInterval::createFromDateString($interval);
	}

	/**
	 * Vytvoří objekt DateTime.
	 *
	 * Vstupním parametrem může být instance této třídy, objektu DateTime nebo string
	 * zpracovatelný pomocí strtotime().
	 *
	 * @param mixed $time Z parametru jít vytvořit objekt DateTime.
	 *
	 * @return \DateTime
	 */
	private function _createDateTimeObject($time)
	{
		if ($time instanceof self) {
			$result = $time->getDateTime();
		} elseif ($time instanceof \DateTime) {
			$result = $time;
		} else {
			$result = new \DateTime($time);
		}
		return $result;
	}

	/**
	 * Vrátí vlastní instanci s aktuálním časem.
	 *
	 * @return Date
	 */
	public static function now()
	{
		return new self('now');
	}

	/**
	 * Vrátí vastní instanci objektu s časem z unix timestamp.
	 *
	 * @param integer $unixTimestamp Čas ve formátu unix timestamp.
	 *
	 * @return Date
	 */
	public static function fromUnixTimestamp($unixTimestamp)
	{
		return new self(\sprintf('@%s', $unixTimestamp));
	}

	/**
	 * Umožňuje vytvoření hluboké kopie.
	 */
	public function __clone()
	{
		$this->_datetime = clone $this->_datetime;
	}

	/**
	 * Vrátí instanci vnitřního objektu DateTime.
	 *
	 * @return \DateTime
	 */
	public function getDateTime()
	{
		return $this->_datetime;
	}

	/**
	 * Vrátí textovou reprezentaci času.
	 *
	 * @return string
	 */
	public function __toString()
    {
        return $this->getDateTime()->format(self::DATE_TIME);
    }
}
