<?php

namespace Dalten\Util;

/**
 * Kolekce helper funkcí pro práci s poli.
 *
 * @category Dalten
 * @package  Util
 */
class ArrayHelper
{
	/**
	 * Projde všechny prvky pole a vrátí hodnotu jejich vlastnosti, určené parametrem.
	 *
	 * <code>
	 * $data = array(
	 *     (object) array('xx' => 15),
	 *     (object) array('xx' => 19),
	 *     (object) array('xx' => 19)
	 * );
	 *
	 * Dalten\Util\ArrayHelper::collectProperties($data, 'xx');
	 * // vrátí array(15, 19, 19)
	 * </code>
	 *
	 * @param array  $items        Pole objektů, ze kterých se mají získat hodnoty vlastností.
	 * @param string $propertyName Název vlastnosti, která se má získat.
	 *
	 * @return array Pole získaných hodnot vlastností.
	 */
	public static function collectProperties(array $items, $propertyName)
	{
		$filter = function ($item) use ($propertyName) {
			return $item->{$propertyName};
		};

		return \array_map($filter, $items);
	}

	/**
	 * Vytvoří pole objektů indexované podle určené vlastnosti objektu.
	 *
	 * Pozor pokud je hodnota vlastnoti v objektech duplicitní uloží se posledí v řadě.
	 *
	 * <code>
	 * $data = array(
	 *      (object),
	 *      (object),
	 *      (object),
	 * );
	 * Dalten\Util\ArrayHelper::buildIndexedCollection($data, 'id');
	 * </code>
	 *
	 * @param array  $collection   Pole objektů.
	 * @param string $propertyName Název vlastnosti podle které budeme vytvářet index.
	 *
	 * @return array Nová kolekce objeků klíčovaná podle hodnoty konkrétní vlastnosti.
	 */
	public static function buildIndexedCollection(array $collection, $propertyName)
	{
		$result = array();
		foreach ($collection as $object) {
			$result[$object->{$propertyName}] = $object;
		}
		unset($object);

		return $result;
	}

	/**
	 * Vytvoří nové pole klíčované podle hodnoty v poli polí.
	 *
	 * <code>
	 * $data = array(
	 *      array('key' => 2),
	 *      array('key' => 8),
	 *      array('key' => 666),
	 * );
	 * Dalten\Util\ArrayHelper::buildIndexedCollection($data, 'id');
	 * // vrátí array(2 => array(...), 3 => array(...), 666 => array(...));
	 * </code>
	 *
	 * @param array  $array Pole podle kterého budeme vytvřet nové
	 * @param string $key   Název klíče v poli.
	 *
	 * @return array Nové indexované pole podle určeného klíče.
	 */
	public static function buildArrayByKey(array $array, $key)
	{
		$result = array();
		foreach ($array as $items) {
			$result[$items[$key]] = $items;
		}
		unset($items);

		return $result;
	}

	/**
	 * Vytvoří dvou úrovňové pole kde rekurzivně spojí klíče do textového řetězce.
	 *
	 * @param array  $data      Pole které chceme zploštit.
	 * @param string $keyPrefix Prefix klíčů (používá se hlavně při rekurzivním volání).
	 *
	 * <code>
	 * $data = array(
	 *     'param1' => array(
	 *         'param1value1' => 1,
	 *         'param1value2' => 2
	 *     )
	 * );
	 * self::recursiveToFilterNames($data);
	 * >> array('param1' => 'on', 'param1[param1value1]' => 1, 'param1[param1value2]' => 2)
	 * </code>
	 *
	 * @return array
	 */
	public static function recursiveToFilterNames(array $data, $keyPrefix = '')
	{
		$result = array();
		foreach ($data as $key => $values) {
			$arrayKey = $key;
			if (!empty($keyPrefix)) {
				$arrayKey = sprintf('%s[%s]', $keyPrefix, $key);
			}
			if (!is_array($values)) {
				$result[$arrayKey] = $values;
			} else {
				$result[$arrayKey] = 'on';
				$result = array_replace_recursive($result, self::recursiveToFilterNames($values, $arrayKey));
			}
		}

		return $result;
	}

	/**
	 * Rekurzivní ksort.
	 *
	 * Bacha polse se předává referencí jako je to u klasické ksort fuknce.
	 *
	 * @param array $array     Pole které chceme rekurzivně seřadit podle ksort funkce.
	 * @param int   $sortFlags Radící metoda.
	 *
	 * @return bool
	 */
	public static function ksortRecursive(&$array, $sortFlags = SORT_REGULAR)
	{
		if (!is_array($array)) {
			return false;
		}
		ksort($array, $sortFlags);
		foreach ($array as &$arr) {
			self::ksortRecursive($arr, $sortFlags);
		}

		return true;
	}

	/**
	 * Převede pole (čísel) na string, kde pozice v znaku v řetězci je hodnota v poli a samotný znak je 1 pokud
	 * je daná hodnota v poli nastavena. Jinak je 0.
	 *
	 * @param int[] $values Pole hodnot.
	 *
	 * @return string Řetězcová reprzentace pole viz popis metody.
	 */
	public static function convertToMultiOptionString(array $values)
	{
		foreach ($values as $index => $value) {
			if ($value < 0) {
				unset($values[$index]);
			}
		}

		if (count($values) === 0) {
			return '';
		}
		$return = \str_repeat(0, \max($values));
		// $stringValue je prazdne, pokud prijde nula [array(0)], tedy ji nastavime a frcime dal.
		if ($return === '') {
			return '1';
		}
		foreach ($values as $value) {
			$return[$value] = 1;
		}

		return $return;

	}
}
