<?php

namespace Dalten\WebBundle\Breadcrumbs\Common\Builder;

use Dalten\Util\StringHelper;
use Dalten\WebBundle\Filter\FilterWithAddressInterface;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\BreadcrumbInterface;

/**
 * Pomocný tvořič drobků na kraj a region.
 */
class CountyAndRegionHelper
{
	/**
	 * Mapa ID kraje (klíč) na slug kraje (hodnota).
	 *
	 * @var array
	 */
	protected $_countyIdToSlugMap;

	/**
	 * Mapa ID kraje (klíč) na label kraje (hodnota).
	 *
	 * @var array
	 */
	protected $_countyIdToLabelMap;

	/**
	 * Mapa ID kraje na ID regionu na label regionů.
	 *
	 * @var array
	 */
	protected $_countyToRegionLabelMap;

	/**
	 * Nastavuje konfiguraci.
	 *
	 * Parametrem by mělo být pole s klíči: countyIdToSlugMap, countyIdToLabelMap.
	 *
	 * @param array $countyIdToSlugMap Mapa ID kraje (klíč) na slug kraje (hodnota).
	 * @param array $countyToRegionLabelMap Mapa ID kraje na ID regionu na label regionů.
	 */
	public function __construct(array $countyIdToSlugMap, array $countyToRegionLabelMap)
	{
		$this->_countyIdToSlugMap = $countyIdToSlugMap;
		$this->_countyToRegionLabelMap = $countyToRegionLabelMap;
	}

	/**
	 * Vytvoří drobka z vyplněného filtru.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * V routeParams předchozího drobku musí být klíč 'county' se slugem kraje.
	 *
	 * @param FilterWithAddressInterface $filter             Vyplněný filtr s adresami.
	 * @param BreadcrumbInterface        $previousBreadcrumb Předchozí drobek (s krajem).
	 *
	 * @internal param string $routeName Název tvořené routy.
	 * @internal param array $routeParams Parametry routy (bude přidán klíč county).
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromFilter(FilterWithAddressInterface $filter, BreadcrumbInterface $previousBreadcrumb)
	{
		$countyIds = $regionIds = array();
		foreach ($filter->getAddressFilters() as $addressFilter) {
			$countyIds[] = $addressFilter->county_code;
			$regionIds = array_merge($regionIds, $addressFilter->region_codes);
		}

		$regionId = current($regionIds);

		return $this->getBreadcrumb($previousBreadcrumb, $countyIds, $regionId);
	}

	/**
	 * Vytvoří model drobečku dle předaného předchozího drobku kraje a regionu a vrátí jej.
	 *
	 * V routeParams předchozího drobku musí být klíč 'county' se slugem kraje.
	 *
	 * @param BreadcrumbInterface $previousBreadcrumb Předchozí drobek.
	 * @param array               $countyCodes        Id krajů v configu.
	 * @param int                 $regionCode         Id regionu v configu.
	 *
	 * @return null|BreadcrumbInterface Drobek nebo null pokud jej nelze vytvořit.
	 */
	public function getBreadcrumb(BreadcrumbInterface $previousBreadcrumb, array $countyCodes, $regionCode)
	{
		$result = $this->_getCorrectRegionAndCountyCodes($previousBreadcrumb, $countyCodes, $regionCode);

		if (empty($result['county_code']) || empty($result['region_code'])) {
			return null;
		}

		$info = $this->_getBreadcrumbData($result['county_code'], $result['region_code']);

		if (empty($info)) {
			return null;
		}

		$params = array_merge(
			$previousBreadcrumb->getRouteParams(),
			array(
				'region' => $info['region_slug'],
				'county' => $info['county_slug']
			)
		);

		$routeName = $previousBreadcrumb->getRouteName() . 'AndRegion';

		return new Breadcrumb($info['label'], $routeName, $params);
	}

	/**
	 * Převede pole možných kódů krajů a kód regionu na dvojici regionu a kraje, kde region je normalizovaný
	 * a kraj je ten kraj, který obsahuje daný region.
	 *
	 * @param BreadcrumbInterface $previousBreadcrumb Předchozí drobek.
	 * @param array               $countyCodes        Pole kódů krajů.
	 * @param int                 $regionCode         Kód regionu.
	 *
	 * @return array Pole s klíči region_code a county_code.
	 */
	private function _getCorrectRegionAndCountyCodes(BreadcrumbInterface $previousBreadcrumb, array $countyCodes,
		$regionCode)
	{
		$routeParams = $previousBreadcrumb->getRouteParams();
		if (!isset($routeParams['county'])) {
			return array('region_code' => $regionCode, 'county_code' => null);
		}
		$previousCountySlug = $routeParams['county'];
		$parentCountyCode = null;

		foreach ($countyCodes as $countyCode) {
			// Najdeme kód kraje, do kterého daný region patří.
			if (isset($this->_countyToRegionLabelMap[$countyCode][$regionCode])) {
				if ($previousCountySlug === $this->_countyIdToSlugMap[$countyCode]) {
					$parentCountyCode = $countyCode;
					break;
				}
			}
		}

		return array('region_code' => $regionCode, 'county_code' => $parentCountyCode);
	}

	/**
	 * Získá z kódu kraje a regionu data pro vytvoření drobečku.
	 *
	 * Výsledné pole je buď prázdné (nepodařilo se) a nebo má klíče:
	 *  - county_code, region_code, region_slug, county_slug, label
	 *
	 * @param int $countyCode Id kraje.
	 * @param int $regionCode Id regionu (již převedeno na správný tvar).
	 *
	 * @return array Pole informací pro vytvoření drobku a nebo prázdné pole.
	 */
	private function _getBreadcrumbData($countyCode, $regionCode)
	{
		if (empty($countyCode) || empty($regionCode)) {
			return array();
		}

		$label = $this->_countyToRegionLabelMap[$countyCode][$regionCode];
		$countySlug = isset($this->_countyIdToSlugMap[$countyCode])
			? $this->_countyIdToSlugMap[$countyCode] : null;

		if (empty($countySlug)) {
			return array();
		}

		return array(
			'county_code' => $countyCode,
			'region_code' => $regionCode,
			'region_slug' => StringHelper::slugify($label),
			'county_slug' => $countySlug,
			'label' => $label
		);
	}
}
