<?php

namespace Dalten\WebBundle\Breadcrumbs\Listing\Builder;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Entity\Listing;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\BreadcrumbInterface;

/**
 * Tvořič drobečků na základě typu nemovitosti a zakázky.
 */
class ListingAndAdvertType implements BuilderInterface
{
	/**
	 * Konfigurace.
	 *
	 * @var array
	 */
	private $_config;

	private $_listingTypeBuilder;

	/**
	 * Nastavuje konfiguraci.
	 *
	 * Konfigurační pole by mělo mít klíče routeName a pole advertTypeMap, ve kterých je každý prvek
	 * pole s klíči label a routeParams.
	 *
	 * @param BuilderInterface $listingTypeBuilder Builder na drobky typu nabídky.
	 * @param array            $config Pole nastavení.
	 */
	public function __construct(BuilderInterface $listingTypeBuilder, array $config)
	{
		$this->_config = $config;
		$this->_listingTypeBuilder = $listingTypeBuilder;
	}

	/**
	 * Vytvoří drobka z vyplněného modelu nabídky.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param Listing $listing Model nabídky.
	 *
	 * @return Breadcrumb|null Drobek nebo null.
	 */
	public function getBreadcrumbFromListing(Listing $listing)
	{
		$previousBreadcrumb = $this->_listingTypeBuilder->getBreadcrumbFromListing($listing);

		if (empty($previousBreadcrumb)) {
			return null;
		}
		return $this->_getBreadcrumb($previousBreadcrumb, $listing->advert_type);
	}

	/**
	 * Vytvoří drobka z vyplněného filtru nabídek.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return Breadcrumb|null Drobek nebo null.
	 */
	public function getBreadcrumbFromFilter(ListingFilter $filter)
	{
		$filterData = $filter->toArray();
		$advertType = empty($filterData['advert_type']) ? null : $filterData['advert_type'];
		$previousBreadcrumb = $this->_listingTypeBuilder->getBreadcrumbFromFilter($filter);

		if (empty($previousBreadcrumb)) {
			return null;
		}

		return $this->_getBreadcrumb(
			$previousBreadcrumb, is_array($advertType) ? current($advertType) : $advertType
		);
	}

	/**
	 * Vytvoří model drobečku dle předaného typu inzerátu a drobku na typ nemovitosti a vrátí jej.
	 *
	 * @param BreadcrumbInterface $listingTypeBreadcrumb Drobek s typem nemovitosti.
	 * @param int                 $advertType            Id typu inzerátu.
	 *
	 * @return null|Breadcrumb Drobek nebo null pokud jej nelze vytvořit.
	 */
	private function _getBreadcrumb(BreadcrumbInterface $listingTypeBreadcrumb, $advertType)
	{
		$currentAdvertTypeConfig = array();

		foreach ($this->_config['advertTypeMap'] as $typeConfig) {
			if (in_array($advertType, $typeConfig['acceptedValues'])) {
				$currentAdvertTypeConfig = $typeConfig;
				break;
			}
		}

		if (empty($currentAdvertTypeConfig)) {
			return null;
		}

		$params = array_merge($listingTypeBreadcrumb->getRouteParams(), $currentAdvertTypeConfig['routeParams']);

		return new Breadcrumb($currentAdvertTypeConfig['label'], $this->_config['routeName'], $params);
	}
}
