<?php

namespace Dalten\WebBundle\Breadcrumbs\Listing\Builder;

use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Entity\Listing;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\BreadcrumbInterface;
use Dalten\WebBundle\Breadcrumbs\Common\Builder\CountyHelper;

/**
 * Tvořič drobečků na základě předchozího drobečku a kraje.
 */
class PreviousBreadcrumbAndCounty implements BuilderInterface
{
	/**
	 * Konfigurace.
	 *
	 * @var array
	 */
	protected $_config;

	/**
	 * Pole tvořičů předchozích drobků.
	 *
	 * @var BuilderInterface[]
	 */
	protected $_previousBreadcrumbBuilders;

	/**
	 * Instance obecného builderu (helperu) na kraje.
	 *
	 * @var CountyHelper
	 */
	protected $_helper;

	/**
	 * Nastavuje konfiguraci.
	 *
	 * Pole tvořičů předchozích drobků by mělo být seřazeno tak, aby byl ten nejpřesnější drobek první, atd.
	 *
	 * V config se očekávají klíče: countyIdToSlugMap, allCountiesLabel, allCountiesSlug. Všechny jsou nepovinné,
	 * při jejich uvedení se builder pokusí vytvořit drobek pro všechny kraje (jsou-li všechny vybrány).
	 *
	 * @param BuilderInterface[] $previousBreadcrumbBuilders Pole předchozích tvořičů drobků.
	 * @param CountyHelper       $countyHelper               Instance obecného builderu (helperu) na kraje.
	 * @param array              $config                     Pole nastavení.
	 */
	public function __construct(array $previousBreadcrumbBuilders, CountyHelper $countyHelper, array $config)
	{
		$this->_config = $config;
		$this->_previousBreadcrumbBuilders = $previousBreadcrumbBuilders;
		$this->_helper = $countyHelper;
	}

	/**
	 * Vytvoří drobka z vyplněného modelu nabídky.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param Listing $listing Model nabídky.
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromListing(Listing $listing)
	{
		$previousBreadcrumb = null;
		foreach ($this->_previousBreadcrumbBuilders as $builder) {
			$previousBreadcrumb = $builder->getBreadcrumbFromListing($listing);
			if ($previousBreadcrumb !== null) {
				break;
			}
		}

		if ($previousBreadcrumb === null) {
			return null;
		}

		return $this->_getBreadcrumb($previousBreadcrumb, $listing->listing_locality_county_code);
	}

	/**
	 * Vytvoří drobka z vyplněného filtru nabídek.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromFilter(ListingFilter $filter)
	{
		$previousBreadcrumb = null;
		foreach ($this->_previousBreadcrumbBuilders as $builder) {
			$previousBreadcrumb = $builder->getBreadcrumbFromFilter($filter);
			if ($previousBreadcrumb !== null) {
				break;
			}
		}

		if ($previousBreadcrumb === null) {
			return null;
		}

		$allCountiesBreadcrumb = $this->_createAllCountiesBreadcrumb($previousBreadcrumb, $filter);
		if ($allCountiesBreadcrumb) {
			return $allCountiesBreadcrumb;
		}

		/** @var AddressFilter $addressFilter */
		$addressFilter = current($filter->getAddressFilters());

		if (empty($addressFilter) || empty($addressFilter->county_code)) {
			return null;
		}

		return $this->_getBreadcrumb($previousBreadcrumb, $addressFilter->county_code);
	}

	/**
	 * Vytvoří model drobečku dle předaného předchozího drobku a kraje a vrátí jej.
	 *
	 * @param BreadcrumbInterface $previousBreadcrumb Předchozí drobek.
	 * @param int                 $countyCode         Id kraje v adresní DB.
	 *
	 * @return null|BreadcrumbInterface Drobek nebo null pokud jej nelze vytvořit.
	 */
	private function _getBreadcrumb(BreadcrumbInterface $previousBreadcrumb, $countyCode)
	{
		return $this->_helper->getBreadcrumb(
			$countyCode,
			$previousBreadcrumb->getRouteName() . '_and_county',
			$previousBreadcrumb->getRouteParams()
		);
	}

	/**
	 * Pokusí se vytvořit drobka pro všechny kraje (ceska-republika).
	 *
	 * Drobek je vytvořen pokud jsou splněny obě následující podmínky:
	 *            Jsou vybrány všechny definované kraje.
	 *            V $config jsou klíče: allCountiesLabel a allCountiesSlug.
	 *
	 * @param BreadcrumbInterface $previousBreadcrumb Předchozí drobek.
	 * @param ListingFilter       $filter             Filtr nabídek.
	 *
	 * @return Breadcrumb|null Drobek pro všechny kraje a nebo null, nejde-li vytvořit.
	 */
	private function _createAllCountiesBreadcrumb(BreadcrumbInterface $previousBreadcrumb, ListingFilter $filter)
	{
		if (!isset($this->_config['countyIdToSlugMap'])) {
			return null;
		}

		$counties = array();

		foreach ($filter->getAddressFilters() as $addressFilter) {
			$counties[] = $addressFilter->county_code;
		}

		sort($counties);
		$allCounties = array_keys($this->_config['countyIdToSlugMap']);
		sort($allCounties);

		if ($counties == $allCounties && isset($this->_config['allCountiesLabel'])
			&& isset($this->_config['allCountiesSlug'])
		) {

			$params = array_merge(
				$previousBreadcrumb->getRouteParams(),
				array('county' => $this->_config['allCountiesSlug'])
			);

			$routeName = $previousBreadcrumb->getRouteName() . '_and_county';

			return new Breadcrumb($this->_config['allCountiesLabel'], $routeName, $params);
		}

		return null;
	}
}
