<?php

namespace Dalten\WebBundle\Filter;

use Dalten\WebBundle\Filter\Exception\PropertyDoesNotExist;

/**
 * Základní obálka pro práci s daty filtru nemovitostí.
 *
 * @package Dalten\WebBundle\Filter
 */
class Base implements FilterInterface
{
	private $_data = array();

	/**
	 * Nastaví data do filtru.
	 *
	 * @param array $data Data pro filtr.
	 */
	public function __construct(array $data)
	{
		$this->_data = $data;
	}

	/**
	 * {@inheritdoc}
	 */
	public function __get($name)
	{
		if (false === array_key_exists($name, $this->_data)) {
			throw new PropertyDoesNotExist(
				\sprintf(
					'Položka "%s" ve filtru %s neexistuje',
					$name,
					\get_class($this)
				)
			);
		}

		return $this->_data[$name];
	}

	/**
	 * {@inheritdoc}
	 */
	public function __isset($name)
	{
		return isset($this->_data[$name]);
	}

	public function toArray()
	{
		return $this->_data;
	}

	/**
	 * Normalizuje data dle předané definice.
	 *
	 * @param array $propertyDefinition Definice dat (pole:
	 * @param array $data
	 *
	 * @return array
	 */
	protected function _normalizeData(array $propertyDefinition, array $data)
	{
		$normalizedData = array();

		foreach ($propertyDefinition as $propertyName => $valueType) {
			$value = isset($data[$propertyName]) ? $data[$propertyName] : null;
			switch ($valueType) {
				case 'int':
					$normalizedData[$propertyName] = (int) $value;
					break;
				case 'float':
					$normalizedData[$propertyName] = (float) $value;
					break;
				case 'multiOption':
					$normalizedData[$propertyName] = \array_map('intval', (array) $value);
					break;
				case 'string':
					$normalizedData[$propertyName] = (string) $value;
					break;
				case 'array':
					$normalizedData[$propertyName] = (array) $value;
					break;
				case 'int[]':
					$normalizedData[$propertyName] = array_map('intval', (array) $value);
					break;
				case 'bool':
					$normalizedData[$propertyName] = (bool) $value;
			}
		}

		return $normalizedData;
	}
}
