<?php
namespace Dalten\WebBundle\Processor\ListingListData;

use Dalten\WebBundle\Filter\Converter\ListingListInterface;
use Dalten\WebBundle\Processor\UriParamsToQueryConverter;
use Dalten\WebBundle\Filter\ListingFilter;
use Symfony\Component\Form\Form;
use Symfony\Component\HttpFoundation\Request;

/**
 * Builder, zapouzřují nastavení všech potřebných argumentů pro vytvoření instance ListingListData\Data.
 *
 * Každá instance builderu slouží k vytvoření pouze jedné instance Dat. DI musí být nastaveno tak, aby pokaždé
 * vracelo novou instanci.
 */
class Builder
{
	/**
	 * Jedná se o výpis zlevněných nemovitostí?
	 *
	 * @var bool
	 */
	private $_discounted = false;

	/**
	 * Jedná se o výpis zahraničních nemovitostí?
	 *
	 * @var bool
	 */
	private $_foreign = false;

	/**
	 * Jedná se o výpis oblíbených nemovitostí?
	 *
	 * @var bool
	 */
	private $_favorite = false;

	/**
	 * Jedná se o výpis nemovitostí s dnem otevřených dveří?
	 *
	 * @var bool
	 */
	private $_openDay = false;

	/**
	 * Instance filtru.
	 *
	 * @var ListingFilter
	 */
	private $_filter;

	/**
	 * Pole, podle kterého řadíme výsledky.
	 *
	 * @var string
	 */
	protected $_orderByField = Data::ORDER_BY_PUBLISHED_DATE;

	/**
	 * Řadíme vzestupně?
	 *
	 * @var bool
	 */
	protected $_orderAscending = false;

	/**
	 * Pole id oblíbených nabídek.
	 *
	 * @var int[]
	 */
	private $_favoriteListingIds = array();

	/**
	 * Převaděč mezi URL parametry a filtrem.
	 *
	 * @var UriParamsToQueryConverter
	 */
	private $_uriParamsConverter;

	/**
	 * Číslo stránky.
	 *
	 * @var int
	 */
	private $_pageNumber = 1;

	/**
	 * Název parametru, určujícího číslo stránky.
	 *
	 * @var string
	 */
	private $_pageVariableName;

	/**
	 * Převaděč mezi query params a filtrem.
	 *
	 * @var ListingListInterface
	 */
	private $_filterConverter;

	/**
	 * Nastavení závislostí.
	 *
	 * @param UriParamsToQueryConverter $uriParamsConverter Převaděč mezi URL atributy a query.
	 * @param ListingListInterface      $filterConverter    Převaděč mezi query params a filtrem.
	 * @param string                    $pageVariableName   Název parametru, určujícího číslo stránky.
	 */
	public function __construct(UriParamsToQueryConverter $uriParamsConverter, ListingListInterface $filterConverter,
		$pageVariableName)
	{
		$this->_uriParamsConverter = $uriParamsConverter;
		$this->_pageVariableName = $pageVariableName;
		$this->_filterConverter = $filterConverter;
	}

	/**
	 * Nastaví filtr.
	 *
	 * @param ListingFilter $filter Instance filtru.
	 *
	 * @return $this
	 */
	public function setFilter(ListingFilter $filter)
	{
		$this->_filter = $filter;

		return $this;
	}

	/**
	 * Nastavuje, zda se jedná o výpis zlevněných nabídek.
	 *
	 * @param boolean $discounted Jedná se o výpis zlevněných nabídek?
	 *
	 * @return $this
	 */
	public function setIsDiscounted($discounted = true)
	{
		$this->_discounted = (bool) $discounted;

		return $this;
	}

	/**
	 * Nastavuje, zda se jedná o výpis oblíbených nabídek.
	 *
	 * @param boolean $favorite Jedná se o výpis oblíbených nabídek?
	 *
	 * @return $this
	 */
	public function setIsFavorite($favorite = true)
	{
		$this->_favorite = (bool) $favorite;

		return $this;
	}

	/**
	 * Nastavuje, zda se jedná o výpis zahraničních nabídek.
	 *
	 * @param boolean $foreign Jedná se o výpis zahraničních nabídek?
	 *
	 * @return $this
	 */
	public function setIsForeign($foreign = true)
	{
		$this->_foreign = (bool) $foreign;

		return $this;
	}

	/**
	 * Nastavuje, zda se jedná o výpis nabídek se dnem otevřených dveří.
	 *
	 * @param boolean $openDays Jedná se o výpis nabídek se dnem otevřených dveří?
	 *
	 * @return $this
	 */
	public function setIsOpenDay($openDays = true)
	{
		$this->_openDay = (bool) $openDays;

		return $this;
	}

	/**
	 * Nastavuje číslo stránky.
	 *
	 * @param int $pageNumber Číslo stránky (1+).
	 *
	 * @return $this
	 */
	public function setPageNumber($pageNumber)
	{
		$this->_pageNumber = (int) $pageNumber;

		return $this;
	}

	/**
	 * Nastavuje řazení výpisu dle data vložení nabídky ve směru určeném parametrem.
	 *
	 * @param bool $asc Řadit vzestupně?
	 *
	 * @return $this
	 */
	public function orderByPublishedDate($asc = true)
	{
		$this->_orderByField = Data::ORDER_BY_PUBLISHED_DATE;
		$this->_orderAscending = $asc;

		return $this;
	}

	/**
	 * Nastavuje řazení výpisu dle ceny nabídky ve směru určeném parametrem.
	 *
	 * @param bool $asc Řadit vzestupně?
	 *
	 * @return $this
	 */
	public function orderByPrice($asc = true)
	{
		$this->_orderByField = Data::ORDER_BY_PRICE;
		$this->_orderAscending = $asc;

		return $this;
	}

	/**
	 * Nastavuje seznam ID oblíbených nabídek.
	 *
	 * @param array $favoriteListingIds Pole ID oblíbených nabídek.
	 *
	 * @return $this
	 */
	public function setFavoriteListingIds(array $favoriteListingIds)
	{
		$this->_favoriteListingIds = $favoriteListingIds;

		return $this;
	}

	/**
	 * Nastaví builder podle parametrů requestu.
	 *
	 * @param Request $request             Instance requestu.
	 * @param Form    $filterForm          Instance formu filtru.
	 * @param array   $favouriteListingIds Pole ID oblíbených nabídek.
	 *
	 * @return $this
	 */
	public function buildFromRequest(Request $request, Form $filterForm, array $favouriteListingIds = array())
	{

		if ($request->attributes->get('_foreignListings')) {
			$this->setIsForeign();
		}

		if ($request->attributes->get('_favourites')) {
			$this->setIsFavorite();
		}

		if ($request->attributes->get('has_open_day')) {
			$this->setIsOpenDay();
		}

		if ($request->attributes->get('discount_duration')) {
			$this->setIsDiscounted();
		}

		$priceOrder = $request->query->get('order_by_price');
		if (isset($priceOrder)) {
			$this->orderByPrice($priceOrder);
		}

		$publishedOrder = $request->query->get('order_by_published_date');
		if (isset($publishedOrder)) {
			$this->orderByPublishedDate($publishedOrder);
		}

		$this->setFavoriteListingIds($favouriteListingIds);

		$filterData = (array) $filterForm->getNormData();
		$requestAttributes = $request->attributes->all();
		if ($requestAttributes) {
			$filterData = array_replace_recursive(
				$filterData, $this->_uriParamsConverter->convertParamsToQuery($requestAttributes)
			);
		}

		$this->setPageNumber((int) $request->query->get($this->_pageVariableName, 1));

		$filter = $this->_filterConverter->createFilterFromArray($filterData);
		$this->setFilter($filter);

		if ($filter->has_open_day) {
			$this->setIsOpenDay();
		}

		return $this;
	}

	/**
	 * Vrátí naplněný objekt ListigListData\Data.
	 *
	 * @return Data
	 */
	public function getData()
	{
		return new Data(
			$this->_filter,
			$this->_discounted,
			$this->_foreign,
			$this->_favorite,
			$this->_openDay,
			$this->_favoriteListingIds,
			$this->_pageNumber,
			$this->_orderByField,
			$this->_orderAscending
		);
	}
}
