<?php
namespace Dalten\WebBundle\Processor;

use Dalten\WebBundle\Entity\Listing;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Procesor na generování titulků nabídek.
 */
class ListingTitleGenerator
{
	/**
	 * Nastavuje translator pro překládání textů.
	 *
	 * @param TranslatorInterface $translator Instance translatoru.
	 */
	public function __construct(TranslatorInterface $translator)
	{
		$this->_translator = $translator;
	}

	/**
	 * Vytvoří a vrátí titulek nabídky.
	 *
	 * @param Listing $listing Instance nabídky.
	 *
	 * @return string Vygenerovaný titulek nabídky.
	 */
	public function getTitleForListing(Listing $listing)
	{
		$title = array();
		if ($listing->advert_type == 1) {
			$title[] = $this->_translator->trans('Prodej');
		} elseif ($listing->advert_type == 2) {
			$title[] = $this->_translator->trans('Pronájem');
		} else {
			$title[] = $this->_translator->trans('Nabídka');
		}

		switch ($listing->listing_type) {
			case 1:
				$title[] = $this->_translator->trans('zemědělského objektu');
				break;
			case 2:
				$title[] = $this->_translator->trans('komerčního objektu');
				$subtype = $this->_getCommercialObjectSubtype($listing->listing_commercial_kind);
				if ($subtype) {
					$title[] = $subtype;
				}
				break;
			case 3:
				$title[] = $this->_translator->trans('pozemku');
				break;
			case 4:
				$title[] = $this->_translator->trans('bytu');
				$subtype = $this->_getFlatSubtype($listing->listing_flat_kind);
				if ($subtype) {
					$title[] = $subtype;
				}
				break;
			case 5:
				$title[] = $this->_translator->trans('historického objektu');
				break;
			case 6:
				$title[] = $this->_translator->trans('domu');
				break;
			case 7:
				switch ($listing->listing_hotel_kind) {
					case 1:
						$title[] = $this->_translator->trans('hotelu');
						break;
					case 2:
						$title[] = $this->_translator->trans('penzionu');
						break;
					case 3:
						$title[] = $this->_translator->trans('restaurace');
						break;
					default:
						$title[] = $this->_translator->trans('hotelu / penzionu / restaurace');
						break;
				}
				break;
			case 8:
				$title[] = $this->_translator->trans('nájemního domu');
				break;
			case 9:
				$title[] = $this->_translator->trans('komerčního prostoru');
				$subtype = $this->_getCommercialAreaSubtype($listing->listing_office_kind);
				if ($subtype) {
					$title[] = $subtype;
				}
				break;
			case 10:
				$title[] = $this->_translator->trans('chaty / chalupy');
				break;
			case 11:
				$title[] = $this->_translator->trans('malého objektu');
				break;
		}

		if ($listing->listing_ownership === 1) {
			$title[] = $this->_translator->trans('v osobním vlastnictví');
		} elseif ($listing->listing_ownership === 2) {
			$title[] = $this->_translator->trans('v družstevním vlastnictví');
		}

		$area = $listing->listing_main_area;
		if ($area) {
			$title[] = $area . ' m²';
		}

		if ($listing->listing_locality_country_code === 'CZ' && $listing->listing_locality_county_code === 19) {
			$address = $listing->listing_locality_prague_district;
			if ($address && $listing->listing_locality_municipality_part) {
				$address .= ' - ' . $listing->listing_locality_municipality_part;
			}
		} else {
			$address = $listing->listing_locality_municipality;
		}

		return implode(' ', $title) . ($address ? ', ' . $address : '');
	}

	/**
	 * Vrátí textovou reprezentaci podtypu bytu.
	 *
	 * @param int $flatKind Podtyp bytu.
	 *
	 * @return string Textová reprezentace podtypu bytu.
	 */
	private function _getFlatSubtype($flatKind)
	{
		switch ($flatKind) {
			case 1:
				return $this->_translator->trans('Garsoniéra');
			case 2:
				return $this->_translator->trans('1+kk');
			case 3:
				return $this->_translator->trans('2+kk');
			case 4:
				return $this->_translator->trans('3+kk');
			case 5:
				return $this->_translator->trans('4+kk');
			case 6:
				return $this->_translator->trans('5+kk');
			case 7:
				return $this->_translator->trans('6+kk');
			case 8:
				return $this->_translator->trans('7+kk');
			case 9:
				return $this->_translator->trans('1+1');
			case 10:
				return $this->_translator->trans('2+1');
			case 11:
				return $this->_translator->trans('3+1');
			case 12:
				return $this->_translator->trans('4+1');
			case 13:
				return $this->_translator->trans('5+1');
			case 14:
				return $this->_translator->trans('6+1');
			case 15:
				return $this->_translator->trans('7+1');
			case 16:
				return $this->_translator->trans('Atypický');
			case 17:
				return $this->_translator->trans('Jiný');
		}
		return '';
	}

	/**
	 * Vrátí textovou reprezentaci podtypu komerčních objektů.
	 *
	 * @param int $commercialKind Podtyp komerčního objektu.
	 *
	 * @return string Textová reprezentace podtypu komerčního objektu.
	 */
	private function _getCommercialObjectSubtype($commercialKind)
	{
		switch ($commercialKind) {
			case 1:
				return $this->_translator->trans('Sklad');
			case 2:
				return $this->_translator->trans('Výroba');
			case 3:
				return $this->_translator->trans('Obchodní centrum');
			case 4:
				return $this->_translator->trans('Nákupní středisko');
			case 5:
				return $this->_translator->trans('Průmyslový areál');
			case 6:
				return $this->_translator->trans('Administrativní budova');
			case 7:
				return $this->_translator->trans('Jiný');
		}
		return '';
	}

	/**
	 * Vrátí textovou reprezentaci podtypu komerčních prostorů.
	 *
	 * @param int $officeKind Podtyp komerčního prostoru.
	 *
	 * @return string Textová reprezentace podtypu komerčního prostoru.
	 */
	private function _getCommercialAreaSubtype($officeKind)
	{
		switch ($officeKind) {
			case 1:
				return $this->_translator->trans('Kanceláře');
			case 2:
				return $this->_translator->trans('Obchodní');
			case 3:
				return $this->_translator->trans('Skladovací');
			case 4:
				return $this->_translator->trans('Jiné');
			case 5:
				return $this->_translator->trans('Výrobní');
		}
		return '';
	}
}

