<?php
namespace Dalten\WebBundle\Processor\SeoUrl;

use Dalten\Util\StringHelper;
use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\FilterWithAddressInterface;

/**
 * Helper na vyhledávání slugů s adresou v URL.
 */
class AddressSlugGenerator
{
	/**
	 * Mapa ID kraje (klíč) na jeho slug (hodnota).
	 *
	 * @var array
	 */
	private $_countyIdToSlugMap;

	/**
	 * Mapa ID kraje na ID regionu na label regionu.
	 *
	 * Tj.: array(<id_kraj> => array(<id_region> => <label_region>, ...),...)
	 *
	 * @var array
	 */
	private $_countyIdToRegionIdToLabelMap;

	/**
	 * Pole s konfigurací speciálních slugů.
	 *
	 * Má klíče requirements a slugs.
	 *
	 * Pod requirements je struktura adresy ve tvaru array(<kód kraje> => array(<kódy regionů>), ...).
	 * Pod slugs je pole s klíčem county a volitlně i klíčem region (pokud je).
	 *
	 * @var array
	 */
	private $_specialSlugs;

	/**
	 * Nastavuje configy pro převod adres.
	 *
	 * @param array $countyIdToSlugMap            Mapa ID kraje (klíč) na jeho slug (hodnota).
	 * @param array $countyIdToRegionIdToLabelMap Mapa ID kraje na ID regionu na label regionu.
	 * @param array $specialSlugs                 Pole se speciálními slugy pro složené regiony/kraje.
	 */
	public function __construct(array $countyIdToSlugMap, array $countyIdToRegionIdToLabelMap,
		array $specialSlugs = array())
	{
		$this->_countyIdToSlugMap = $countyIdToSlugMap;
		$this->_countyIdToRegionIdToLabelMap = $countyIdToRegionIdToLabelMap;
		$this->_specialSlugs = $specialSlugs;
	}

	/**
	 * Získá z předaného filtru slugy pro adresní parametry (county a region).
	 *
	 * Pokud se slugy nepodaří najít, vrátí prázdné pole, jinak vrátí pole s nalezenými slugy.
	 *
	 * @param FilterWithAddressInterface $filter
	 *
	 * @return array Pole s nalezenými slugy (klíče county a region[pokud je]) a nebo prázdné pole.
	 */
	public function getAddressUrlParams(FilterWithAddressInterface $filter)
	{
		$addressFilters = $filter->getAddressFilters();

		$specialSlugs = $this->_findSpecialSlugsForFilter($filter);

		if ($specialSlugs) {
			return $specialSlugs;
		}

		/** @var AddressFilter $addressFilter */
		$addressFilter = current($addressFilters);

		if (empty($addressFilter) || count($addressFilters) !== 1) {
			return array();
		}

		$countyId = $addressFilter->county_code;
		$countySlug = $regionSlug = null;

		if (!empty($countyId)) {
			$countyIdToSlugMap = $this->_countyIdToSlugMap;
			$countyIdToRegionIdToLabelMap = $this->_countyIdToRegionIdToLabelMap;

			$countySlug = isset($countyIdToSlugMap[$countyId]) ? $countyIdToSlugMap[$countyId] : null;

			if (count($addressFilter->region_codes) === 1) {
				$regionCodes = $addressFilter->region_codes;
				$regionId = current($regionCodes);
				$regionLabel = isset($countyIdToRegionIdToLabelMap[$countyId][$regionId])
					? $countyIdToRegionIdToLabelMap[$countyId][$regionId] : null;
				if ($regionLabel) {
					$regionSlug = StringHelper::slugify($regionLabel);
				}
			} elseif(count($addressFilter->region_codes) > 1) {
				// Když máme více než jeden region, tak nechceme SEO url
				$countySlug = null;
			}
		}

		return array_filter(array('county' => $countySlug, 'region' => $regionSlug));
	}

	/**
	 * Pokusí se pro filtr najít speciální titulek z konfigurace. Pokud jej nalezne, vrátí jeho title_parts,
	 * jinak vrátí prázdné pole.
	 *
	 * @param FilterWithAddressInterface $filter Vyplněný filtr s omezením lokality.
	 *
	 * @return array Nalezené části titulku nebo prázdné pole.
	 */
	protected function _findSpecialSlugsForFilter(FilterWithAddressInterface $filter)
	{
		$structuredAddress = array();
		foreach ($filter->getAddressFilters() as $addressFilter) {
			$structuredAddress[$addressFilter->county_code] = $addressFilter->region_codes;
		}

		foreach ($this->_specialSlugs as $slugConfig) {
			if ($structuredAddress == $slugConfig['requirements']) {
				return $slugConfig['slugs'];
			}

		}

		return array();
	}
}
