<?php
namespace Dalten\WebBundle\Processor\TitleGenerator;

use Dalten\WebBundle\Filter\FilterWithAddressInterface;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Generátor adresní část titulku.
 *
 * Tento generátor tvoří pouze titulky na základě adresy, neřeší další části titulku.
 *
 * Abstraktní je proto, že jeho jednotlivé implementace budou muset splňovat různé interface.
 */
class AbstractAddress
{
	/**
	 * Převodník kódu kraje na jeho název.
	 *
	 * @var array
	 */
	protected $_countyIdToLabelMap;

	/**
	 * Mapa se strukturou array([id_kraje] => array([id_region] => [název], ...), ...).
	 *
	 * @var array
	 */
	protected $_countyToRegionLabelMap;

	/**
	 * Instance překladače.
	 *
	 * @var \Symfony\Component\Translation\TranslatorInterface
	 */
	protected $_translator;

	/**
	 * Název lokality, jsou-li vybrány všechny kraje.
	 *
	 * @var string
	 */
	protected $_allCountiesTitle;

	/**
	 * Pole zvláštních titulků - nebudou tvořeny běžným způsobem.
	 *
	 * Pole je tvořeno poli s klíči requirements (co má být ve filtru) a title_parts (části titulku,
	 * který má být vrácen).
	 *
	 * @var array
	 */
	protected $_specialTitles = array();

	/**
	 * Nastavuje překladač a konfiguraci adres.
	 *
	 * @param TranslatorInterface $translator             Instance překladače.
	 * @param array               $countyIdToLabelMap     Mapa kódu kraje na jeho název.
	 * @param array               $countyToRegionLabelMap Mapa kódu kraje na kód okres na jeho label.
	 * @param string              $allCountiesTitle       Název lokality, jsou-li vybrány všechny kraje.
	 * @param array               $specialTitles          Pole konfigurace speciálních titulků (viz výše).
	 */
	public function __construct(TranslatorInterface $translator, array $countyIdToLabelMap,
		array $countyToRegionLabelMap, $allCountiesTitle = null, array $specialTitles = array())
	{
		$this->_translator = $translator;
		$this->_countyIdToLabelMap = $countyIdToLabelMap;
		$this->_countyToRegionLabelMap = $countyToRegionLabelMap;
		$this->_allCountiesTitle = $allCountiesTitle;
		$this->_specialTitles = $specialTitles;
	}

	/**
	 * Vytvoří z předaného filtru titulek pro výpis.
	 *
	 * Vytváří pouze adresní část titulku, typ nemovitosti a další části neřeší, tj. je vhodný na začlenění
	 * do specifičtějších generátorů.
	 *
	 * @param FilterWithAddressInterface $filter Vyplněný filtr s adresami.
	 *
	 * @return string Vytvořený titulek pro filtr.
	 */
	protected function _createTitle(FilterWithAddressInterface $filter)
	{
		$parts = array();
		$filterHasRegions = false;
		$counties = array();

		$title = $this->_findSpecialTitleForFilter($filter);

		if ($title) {
			return $title;
		}

		foreach ($filter->getAddressFilters() as $addressFilter) {
			$filterParts = array();
			foreach ($addressFilter->region_codes as $regionCode) {
				if (isset($this->_countyToRegionLabelMap[$addressFilter->county_code][$regionCode])) {
					$filterParts[] = $this->_countyToRegionLabelMap[$addressFilter->county_code][$regionCode];
				}
				$filterHasRegions = true;
			}

			if (empty($filterParts) && isset($this->_countyIdToLabelMap[$addressFilter->county_code])) {
				$counties[] = $addressFilter->county_code;
				$filterParts[] = $this->_countyIdToLabelMap[$addressFilter->county_code];
			}

			$parts = array_merge($parts, array_filter($filterParts));
		}

		$allCounties = array_keys($this->_countyToRegionLabelMap);
		sort($allCounties);
		sort($counties);
		if (isset($this->_allCountiesTitle) && !$filterHasRegions && $counties == $allCounties) {
			return $this->_allCountiesTitle;
		}

		return implode(', ', $parts);
	}

	/**
	 * Pokusí se pro filtr najít speciální titulek z konfigurace. Pokud jej nalezne, vrátí jeho title_parts,
	 * jinak vrátí prázdné pole.
	 *
	 * @param FilterWithAddressInterface $filter Vyplněný filtr s omezením lokality.
	 *
	 * @return array Nalezené části titulku nebo prázdné pole.
	 */
	protected function _findSpecialTitleForFilter(FilterWithAddressInterface $filter)
	{
		$structuredAddress = array();
		foreach ($filter->getAddressFilters() as $addressFilter) {
			$structuredAddress[$addressFilter->county_code] = $addressFilter->region_codes;
		}

		foreach ($this->_specialTitles as $titleConfig) {
			if ($structuredAddress == $titleConfig['requirements']) {
				return $this->_translator->trans($titleConfig['title']);
			}

		}

		return array();
	}
}
