<?php

namespace Dalten\Tests\Util\String;

use Dalten\Util\String;

/**
 * @group dalten
 */
class StringTest extends \PHPUnit_Framework_TestCase
{
	/**
	 * Tests conversion functions.
	 */
	public function testSlugify()
	{
		$this->assertRegExp('~^[a-z0-9]([a-z0-9\-]*[a-z0-9])?$~', String::slugify('žluťoučký kůň příšerně úpěl ďábelské ódy'));
		$this->assertEquals('zlutoucky-kun-priserne-upel-dabelske-ody', String::slugify('?žluťoučký  +  kůň příšerně úpěl ďábelské ódy...'));
		$this->assertEquals('byt-25m2', String::slugify('Byt 25m²'));
	}

	/**
	 * Tests string trimming.
	 */
	public function testCut()
	{
		// Trim on space
		$this->assertEquals('žluťoučký kůň...', $this->checkStringCut('žluťoučký kůň příšerně úpěl ďábelské ódy'));
		// Trim on period
		$this->assertEquals('žluťoučký kůň...', $this->checkStringCut('žluťoučký kůň.Příšerně úpěl ďábelské ódy'));
		// Trim on period and space
		$this->assertEquals('žluťoučký kůň...', $this->checkStringCut('žluťoučký kůň. Příšerně úpěl ďábelské ódy'));
		// Trim on comma
		$this->assertEquals('žluťoučký kůň...', $this->checkStringCut('žluťoučký kůň,příšerně úpěl ďábelské ódy'));
		// Trim on semicolon
		$this->assertEquals('žluťoučký kůň...', $this->checkStringCut('žluťoučký kůň;příšerně úpěl ďábelské ódy'));

		// Word boundary just at the end
		$this->assertEquals('abcdefghijklm...', $this->checkStringCut('abcdefghijklmno pqrst'));
		$this->assertEquals('abcdefghijklm...', $this->checkStringCut('abcdefghijklmn opqrst'));
		$this->assertEquals('abcdefghijklm...', $this->checkStringCut('abcdefghijklm nopqrst'));

		// No word boundaries
		$this->assertEquals('abcdefghijklm...', $this->checkStringCut('abcdefghijklmnopqrstuvwxyz'));

		// Etc as HTML entity
		$this->assertEquals('žluťoučký kůň&hellip;', $this->checkStringCut('žluťoučký kůň příšerně úpěl ďábelské ódy', 14, '&hellip;'));

		// Short
		$shorty = '1234567890';
		$this->assertEquals($shorty, $this->checkStringCut($shorty));
		$this->assertEquals('12...', $this->checkStringCut($shorty, 5));

	}

	/**
	 * Tests word trimming.
	 */
	public function testCutWords()
	{
		$this->assertEquals('žluťoučký kůň příšerně úpěl ďábelské ódy', $this->checkStringWordCut('žluťoučký kůň příšerně úpěl ďábelské ódy', 10));
		$this->assertEquals('žluťo... kůň příšerně úpěl ďábelské ódy', $this->checkStringWordCut('žluťoučký kůň příšerně úpěl ďábelské ódy', 8));
		$this->assertEquals('žl... kůň př... úpěl ďá... ódy', $this->checkStringWordCut('žluťoučký kůň příšerně úpěl ďábelské ódy', 5));

		// Word boundary just at the end
		$this->assertEquals('abcdefghijk... pqrst', $this->checkStringWordCut('abcdefghijklmno pqrst', 14));
		$this->assertEquals('abcdefghijklmn opqrst', $this->checkStringWordCut('abcdefghijklmn opqrst', 14));
		$this->assertEquals('abcdefghijklm nopqrst', $this->checkStringWordCut('abcdefghijklm nopqrst', 14));

		// Etc as HTML entity
		$this->assertEquals('žluťouč&hellip; kůň příšerně úpěl ďábelské ódy', $this->checkStringWordCut('žluťoučký kůň příšerně úpěl ďábelské ódy', 8, '&hellip;'));

		// Short
		$shorty = '12345678';
		$this->assertEquals($shorty, $this->checkStringWordCut($shorty));
		$this->assertEquals('12...', $this->checkStringWordCut($shorty, 5));
	}

	/**
	 * Checks one string.
	 *
	 * @param string $string Input string
	 * @param integer $max Max length
	 * @param string $etc "Etc" definition
	 * @return string
	 */
	private function checkStringWordCut($string, $max = 8, $etc = '...')
	{
		$cut = String::cutWords($string, $max, $etc);

		// &hellip; has length of 1
		$decode = html_entity_decode($cut, ENT_QUOTES, 'UTF-8');
		$cut2 = strtr($decode, array('&hellip;' => '.'));

		$words = preg_split('~\s+~', $string);
		$trimmedWords = preg_split('~\s+~', $cut2);

		$this->assertEquals(count($trimmedWords), count($words));

		foreach ($words as $i => $word) {
			if (mb_strlen($word, 'utf-8') <= $max) {
				$this->assertEquals($word, $trimmedWords[$i], 'Word trimmed even though it was short enough');
			} else {
				$this->assertLessThanOrEqual($max, mb_strlen($trimmedWords[$i], 'utf-8'));
				$this->assertRegExp('~' . ($etc == '&hellip;' ? '...' : $etc) . '$~', $trimmedWords[$i], 'String does not end with ' . $etc);
			}
		}

		return $cut;
	}

	/**
	 * Checks one string.
	 *
	 * @param string $string Input string
	 * @param integer $max Max length
	 * @param string $etc "Etc" definition
	 * @return string
	 */
	private function checkStringCut($string, $max = 16, $etc = '...')
	{
		$cut = String::cut($string, $max, $etc);
		// &hellip; has length of 1
		$cutLength = mb_strlen(strtr(html_entity_decode($cut), array('&hellip;' => '.')), 'utf-8');
		$this->assertLessThanOrEqual($max, $cutLength, 'String is longer');

		if (mb_strlen($string, 'utf-8') <= $max) {
			$this->assertEquals($string, $cut, 'String trimmed even though it was short enough');
		} else {
			$this->assertRegExp('~' . preg_quote($etc) . '$~', $cut, 'String does not end with ' . $etc);
		}
		return $cut;
	}

	/**
	 * @param       $source
	 * @param array $words
	 * @param       $placeholder
	 * @param       $expectedText
	 *
	 * @dataProvider replaceDataProvider
	 */
	public function testReplace($source, array $words, $placeholder, $expectedText)
	{
		$result = String::replace($source, $words, $placeholder);
		$this->assertEquals($expectedText, $result);
	}

	public function replaceDataProvider()
	{
		return array(
			'Lower case'  => array(
				'Frantisek bezel na nakup a koupil sest rohliku.', // $source
				array('nakup', 'sest'), // $words
				'###', // $placeholder
				'Frantisek bezel na ### a koupil ### rohliku.' // $expectedText
			),
			'Upper case'  => array(
				'Frantisek bezel na NAKUP a koupil SEST rohliku.', // $source
				array('nakup', 'sest'), // $words
				'###', // $placeholder
				'Frantisek bezel na ### a koupil ### rohliku.' // $expectedText
			),
			'Mixed case'  => array(
				'Frantisek bezel na NakUP a koupil SeST rohliku.', // $source
				array('nakup', 'sest'), // $words
				'###', // $placeholder
				'Frantisek bezel na ### a koupil ### rohliku.' // $expectedText
			),
		);
	}
}
