<?php
/**
 * Created by PhpStorm.
 * User: Karel
 * Date: 6.2.14
 * Time: 11:45
 */

namespace Dalten\WebBundle\Tests\Processor\TitleGenerator\ListingList;


use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Processor\TitleGenerator\ListingList\ListingSubtype;

class ListingSubtypeTest extends \PHPUnit_Framework_TestCase
{
	private $_codebooks = array(
		'listing_subtype_default' => array(
			2 => 'byty 1+1',
			4 => 'byty 2+1',
			6 => 'byty 3+1',
		),
		'listing_subtype_sale' => array(
			2 => 'bytů 1+1',
			4 => 'bytů 2+1',
			6 => 'bytů 3+1',
		),
		'listing_subtype_lease' => array(
			2 => 'bytu 1+1',
			4 => 'bytu 2+1',
			6 => 'bytu 3+1',
		)
	);
	/** @var  \Symfony\Component\Translation\TranslatorInterface */
	private $_translator;

	protected function setUp()
	{
		$this->_translator = $this->getMockBuilder('Symfony\Component\Translation\TranslatorInterface')
			->getMock();
		$this->_translator->expects($this->any())->method('trans')->will($this->returnArgument(0));
	}

	public function testGeneratorCapitalizesFirstLetter()
	{
		$filter = new ListingFilter(array('listing_type' => 4, 'flat_kind' => 2));

		$this->assertEquals('Byty 1+1', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorReturnsEmptyTitleIfNoListingTypeIsSelected()
	{
		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->never())->method('createTitle');

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), 4, 'flat_kind'
		);

		$filter = new ListingFilter(array());

		$this->assertEquals('', $generator->createTitle($filter));
	}

	public function testGeneratorReturnsEmptyTitleIfIncorrectListingTypeIsSelected()
	{
		$filter = new ListingFilter(array('listing_type' => 3, 'flat_kind' => 2));

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->never())->method('createTitle');

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), 4, 'flat_kind'
		);

		$this->assertEquals('', $generator->createTitle($filter));
	}

	public function testGeneratorReturnsEmptyTitleIfSubtypeIsEmpty()
	{
		$filter = new ListingFilter(array('listing_type' => 4));

		$filter = new ListingFilter(array('listing_type' => 3, 'flat_kind' => 2));

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->never())->method('createTitle');

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), 4, 'flat_kind'
		);

		$this->assertEquals('', $generator->createTitle($filter));
	}

	public function testGeneratorReturnsEmptyTitleIfSubtypePropertyNameIsEmpty()
	{
		$filter = new ListingFilter(array('listing_type' => 4, 'flat_kind' => 2));

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->never())->method('createTitle');

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), 4, ''
		);

		$this->assertEquals('', $generator->createTitle($filter));
	}

	public function testGeneratorReturnsEmptyTitleIfRequiredListingTypeValueIsEmpty()
	{
		$filter = new ListingFilter(array('listing_type' => 4, 'flat_kind' => 2));

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->never())->method('createTitle');

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), null, 'flat_kind'
		);

		$this->assertEquals('', $generator->createTitle($filter));
	}

	public function testBaseGeneratorDoesNotHaveSpecialHandlingForProjects()
	{
		$filter = new ListingFilter(array('listing_type' => 4, 'flat_kind' => 2));
		$projectFilter = new ListingFilter(array('only_projects' => true, 'listing_type' => 4, 'flat_kind' => 2));

		$this->assertEquals($this->_getTitleForFilter($filter), $this->_getTitleForFilter($projectFilter));
	}

	public function testGeneratorCanGroupMultipleSubtypes()
	{
		$filter = new ListingFilter(
			array('listing_type' => array(4), 'flat_kind' => array(2,4), 'advert_type' => 1)
		);

		$this->assertEquals('Prodej bytů 1+1, bytů 2+1', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorCanUsePresetSpecialTitle()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1, 'flat_kind' => 2)
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4), 'flat_kind' => array(2), 'advert_type' => array(1)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, $specialTitles, 4, 'flat_kind'
		);

		$this->assertEquals('Bagr jede', $generator->createTitle($filter));
	}

	public function testGeneratorReturnsNormalTitleIfNoSpecialTitleIsMatched()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1, 'flat_kind' => 2)
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4), 'flat_kind' => array(4)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, $specialTitles, 4, 'flat_kind'
		);

		$this->assertEquals('Prodej bytů 1+1', $generator->createTitle($filter));
	}

	public function testGeneratorCanFillMissingPartsOfSpecialTitle()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'flat_kind' => 2, 'advert_type' => 1),
			array(new AddressFilter(9, array()))
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)
			->will($this->returnValue('kolobezka'));

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, $specialTitles, 4, 'flat_kind'
		);

		$this->assertEquals('Bagr jede kolobezka', $generator->createTitle($filter));
	}

	public function testGeneratorCorrectlyChoosesProperCaseForDifferentAdvertTypes()
	{
		$filter = new ListingFilter(array('listing_type' => 4, 'advert_type' => 1, 'flat_kind' => 2));
		$this->assertEquals('Prodej bytů 1+1', $this->_getTitleForFilter($filter));

		$filter = new ListingFilter(array('listing_type' => 4, 'flat_kind' => 2));
		$this->assertEquals('Byty 1+1', $this->_getTitleForFilter($filter));

		$filter = new ListingFilter(array('listing_type' => 4, 'advert_type' => 2, 'flat_kind' => 2));
		$this->assertEquals('Pronájem bytu 1+1', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorGetsAddressPartFromPassedInAddressGenerator()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'flat_kind' => 2), array(new AddressFilter(7, array()), new AddressFilter(9, array()))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)
			->will($this->returnValue('test'));

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), 4, 'flat_kind'
		);

		$this->assertEquals('Byty 1+1 test', $generator->createTitle($filter));
	}

	private function _getTitleForFilter(ListingFilter $filter)
	{
		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new ListingSubtype(
			$this->_translator, $addressGenerator, $this->_codebooks, array(), 4, 'flat_kind'
		);

		return $generator->createTitle($filter);
	}
}
