<?php

namespace Dalten\WebBundle\Breadcrumbs\Common\Builder;

use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\FilterWithAddressInterface;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\BreadcrumbInterface;

/**
 * Pomocný tvořič drobků na kraj.
 */
class CountyHelper
{
	/**
	 * Mapa ID kraje (klíč) na slug kraje (hodnota).
	 *
	 * @var array
	 */
	protected $_countyIdToSlugMap;

	/**
	 * Mapa ID kraje (klíč) na label kraje (hodnota).
	 *
	 * @var array
	 */
	protected $_countyIdToLabelMap;

	/**
	 * Nastavuje konfiguraci.
	 *
	 * Parametrem by mělo být pole s klíči: countyIdToSlugMap, countyIdToLabelMap.
	 *
	 * @param array $countyIdToSlugMap  Mapa ID kraje (klíč) na slug kraje (hodnota).
	 * @param array $countyIdToLabelMap Mapa ID kraje (klíč) na label kraje (hodnota).
	 */
	public function __construct(array $countyIdToSlugMap, array $countyIdToLabelMap)
	{
		$this->_countyIdToLabelMap = $countyIdToLabelMap;
		$this->_countyIdToSlugMap = $countyIdToSlugMap;
	}

	/**
	 * Vytvoří drobka z vyplněného filtru.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param FilterWithAddressInterface $filter      Vyplněný filtr s adresami.
	 * @param string                     $routeName   Název tvořené routy.
	 * @param array                      $routeParams Parametry routy (bude přidán klíč county).
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromFilter(FilterWithAddressInterface $filter, $routeName,
		array $routeParams = array())
	{
		/** @var AddressFilter $addressFilter */
		$addressFilter = current($filter->getAddressFilters());

		if (empty($addressFilter) || empty($addressFilter->county_code)) {
			return null;
		}

		return $this->getBreadcrumb($addressFilter->county_code, $routeName, $routeParams);
	}

	/**
	 * Vytvoří model drobečku dle předaného předchozího drobku a kraje a vrátí jej.
	 *
	 * @param int    $countyCode  Id kraje v adresní DB.
	 * @param string $routeName   Název tvořené routy.
	 * @param array  $routeParams Parametry routy (bude přidán klíč county).
	 *
	 * @return null|BreadcrumbInterface Drobek nebo null pokud jej nelze vytvořit.
	 */
	public function getBreadcrumb($countyCode, $routeName, array $routeParams = array())
	{
		if (!isset($this->_countyIdToSlugMap[$countyCode]) || !isset($this->_countyIdToLabelMap[$countyCode])) {
			return null;
		}

		$routeParams = array_merge($routeParams, array('county' => $this->_countyIdToSlugMap[$countyCode]));

		return new Breadcrumb($this->_countyIdToLabelMap[$countyCode], $routeName, $routeParams);
	}
}
