<?php

namespace Dalten\WebBundle\Breadcrumbs\Listing\Builder;

use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Entity\Listing;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\BreadcrumbInterface;

/**
 * Drobeček na typ nemovitosti.
 */
class ListingTypeWithSubtype implements BuilderInterface
{
	/**
	 * Konfigurace.
	 *
	 * @var array
	 */
	private $_config;

	/**
	 * Nastavuje konfiguraci.
	 *
	 * Konfigurační pole by mělo obsahovat klíče routeName, valueMap a filterTypeToSubtypeMap,
	 * kde valueMap je pole polí s klíči:
	 *      - label (text drobku)
	 *      - acceptedValues (přípustné hodnoty typu nemovitosti)
	 *      - acceptedSubtypes (přípustné hodnoty podtypů (ve formátu název => [ hodnoty ]
	 *      - filterTypeToSubtypeMap (mapa typu nemovitosti (int) na podtyp (dle názvů ve filtru)
	 *
	 * @param array $config Pole konfigurace. Mělo by obsahovat parametry pro všechny typy nemovitostí.
	 */
	public function __construct(array $config)
	{
		$this->_config = $config;
	}

	/**
	 * Vytvoří drobka z vyplněného modelu nabídky.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param Listing $listing Model nabídky.
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromListing(Listing $listing)
	{
		return $this->_getBreadcrumb(
			array($listing->listing_type),
			$this->_collectSubtypesFromListing($listing)
		);
	}

	/**
	 * Vytvoří drobka z vyplněného filtru nabídek.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromFilter(ListingFilter $filter)
	{
		return $this->_getBreadcrumb(
			$filter->listing_type, $this->_collectSubtypesFromFilter($filter)
		);
	}

	/**
	 * Vytvoří model drobečku dle předaného typu a podtypu nemovitosti a vrátí jej.
	 *
	 * @param array $listingTypes Pole id typu nemovitosti.
	 * @param array $subtypes     Pole s názvy a hodnotami (jako pole) podtypů. ([office_kind => [4])
	 *
	 * @return null|BreadcrumbInterface Drobek nebo null pokud jej nelze vytvořit.
	 */
	private function _getBreadcrumb(array $listingTypes, array $subtypes)
	{
		if (empty($listingTypes) || empty($subtypes)) {
			return null;
		}

		$currentTypeConfig = $fallbackConfig = array();
		sort($listingTypes);
		asort($subtypes);

		foreach ($this->_config['valueMap'] as $typeConfig) {
			$acceptedValues = $typeConfig['acceptedValues'];
			$acceptedSubtypes = $typeConfig['acceptedSubtypes'];
			sort($acceptedValues);
			asort($acceptedSubtypes);
			if ($listingTypes === $acceptedValues && $acceptedSubtypes === $subtypes) {
				$currentTypeConfig = $typeConfig;
				break;
			}

			if ($this->_isArraySubset($listingTypes, $acceptedValues) && $this->_isArraySubset($subtypes, $acceptedSubtypes)) {
				$fallbackConfig = $typeConfig;
			}
		}

		if (empty($currentTypeConfig) && !empty($fallbackConfig)) {
			$currentTypeConfig = $fallbackConfig;
		}

		if (empty($currentTypeConfig)) {
			return null;
		}

		return new Breadcrumb(
			$currentTypeConfig['label'], $this->_config['routeName'], $currentTypeConfig['routeParams']
		);
	}

	/**
	 * Získává z filtru neprázdné podtypy a vrací je jako asicativní pole polí.
	 *
	 * @param ListingFilter $filter Filtr nabídek.
	 *
	 * @return array Pole neprázdných podtypů ([název => [hodnoty]]).
	 */
	private function _collectSubtypesFromFilter(ListingFilter $filter)
	{
		$subtypes = array();

		foreach ($filter->listing_type as $listingType) {
			if (isset($this->_config['filterTypeToSubtypeMap'][$listingType])) {
				$subtypeValue = $filter->{$this->_config['filterTypeToSubtypeMap'][$listingType]};
				$subtypes[$this->_config['filterTypeToSubtypeMap'][$listingType]] = $subtypeValue;
			}
		}

		return array_filter($subtypes);
	}

	/**
	 * Získá z nabídky její podtyp a vrátí jej jako pole, podobné hodnotám filtru.
	 *
	 * @param Listing $listing Model nabídky.
	 *
	 * @return array Pole s podtypem nabídky ([název => [hodnota]]).
	 */
	private function _collectSubtypesFromListing(Listing $listing)
	{
		$subtypes = array();
		$listingType = $listing->listing_type;

		if (isset($this->_config['filterTypeToSubtypeMap'][$listingType])) {
			$subtypes[$this->_config['filterTypeToSubtypeMap'][$listingType]] = array($listing->listing_subtype);
		}

		return array_filter($subtypes);
	}

	/**
	 * Pomocná metoda, zjišťující, zda je pole $subset podmnožinou pole $master.
	 *
	 * @param array $subset Pole, které je testováno, zda je podmnožinou $master.
	 * @param array $master Pole, obsahující všechny přípustné položky.
	 *
	 * @return bool Je $subset podmnožinou $master?
	 */
	private function _isArraySubset(array $subset, array $master)
	{
		foreach ($subset as $property => $value) {
			if (empty($master[$property])) {
				return false;
			}
			if (is_array($value)) {
				foreach ($value as $subValue) {
					if (!in_array($subValue, $master[$property])) {
						return false;
					}
				}
			} elseif (is_numeric($property)) {
				// na číselně indexovaném poli nemůžeme porovnávat přes !=, musíme in_array
				if (!in_array($value, $master)) {
					return false;
				}
			} elseif ($value != $master[$property]) {
				return false;
			}
		}

		return true;
	}
}
