<?php
namespace Dalten\WebBundle\Processor;

use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\Converter\ListingListInterface;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\SeoRoute\FragmentCollection;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Vytváří linky na alternativní filtry v případě, že nebyla nalezena žádná nemovitost.
 */
class EmptyListingResultHandler
{
	/**
	 * Generátor SEO rout.
	 *
	 * @var \Dalten\WebBundle\SeoRoute\FragmentCollection
	 */
	private $_seoRouteMatcher;

	/**
	 * Převaděč filtru nabídek.
	 *
	 * @var \Dalten\WebBundle\Filter\Converter\ListingListInterface
	 */
	private $_filterConverter;

	/**
	 * Pole slugu alternativního linku a jeho labelu.
	 *
	 * @var array
	 */
	private $_linkTypeLabels;

	/**
	 * Překladač.
	 *
	 * @var TranslatorInterface
	 */
	private $_translator;

	/**
	 * Nastavuje závislosti.
	 *
	 * @param FragmentCollection   $seoRouteMatcher Generátor SEO rout.
	 * @param ListingListInterface $filterConverter Převaděč dat filtru nabídek.
	 * @param TranslatorInterface  $translator      Překladač.
	 * @param array                $linkTypeLabels  Pole slugu alternativního linku a jeho labelu.
	 */
	public function __construct(FragmentCollection $seoRouteMatcher,
		ListingListInterface $filterConverter, TranslatorInterface $translator, array $linkTypeLabels)
	{
		$this->_seoRouteMatcher = $seoRouteMatcher;
		$this->_filterConverter = $filterConverter;
		$this->_linkTypeLabels = $linkTypeLabels;
		$this->_translator = $translator;
	}

	/**
	 * Vrátí pole možný alternativních linků na filtr.
	 *
	 * Pole je indexováno slugem linku (region, area, price, subtypes, locality_text, extended) a každý
	 * prvek je pole s klíčí name, parameters a label. Pomocí těchto polí je možné vystavět alternativní routy.
	 *
	 * @param ListingFilter $filter           Vyplněný filtr nabídek.
	 * @param string        $defaultRouteName Routa na defaultní (ne-SEO) výpis.
	 *
	 * @return array Pole nastavení možných alternativních linků (viz popis fce).
	 */
	public function getAlternateFilterLinks(ListingFilter $filter, $defaultRouteName)
	{
		$return = array();
		$return['region'] = $this->_getRouteWithoutRegions($filter, $defaultRouteName);
		$return['area'] = $this->_getRouteWithoutFields(
			$filter, $defaultRouteName, array('main_area_max', 'main_area_min')
		);
		$return['price'] = $this->_getRouteWithoutFields(
			$filter, $defaultRouteName, array('price_max', 'price_min')
		);
		$return['subtype'] = $this->_getRouteWithoutFields(
			$filter, $defaultRouteName,
			array(
				'flat_kind',
				'object_kind_houses',
				'object_kind_cottages',
				'object_kind_smallobjects',
				'commercial_kind',
				'estate_kind',
				'hotel_kind',
				'office_kind'
			)
		);
		$return['locality_text'] = $this->_getRouteWithoutFields($filter, $defaultRouteName, array('locality_text'));
		$return['extended'] = $this->_getRouteWithoutFields(
			$filter, $defaultRouteName,
			array('building_type', 'building_condition', 'ownership', 'floor_min', 'floor_max')
		);

		$return = array_filter($return);

		foreach ($return as $key => $config) {
			$return[$key]['label'] = isset($this->_linkTypeLabels[$key])
				? $this->_translator->trans($this->_linkTypeLabels[$key]) : '';
		}

		return $return;
	}

	/**
	 * Ostraní z filtru omezení na regiony a vrátí konfiguraci upravené routy.
	 *
	 * Pokud regiony nebyly omezeny, vrací null.
	 *
	 * @param ListingFilter $filter           Vyplněný filtr nabídek.
	 * @param string        $defaultRouteName Routa na defaultní (ne-SEO) výpis
	 *
	 * @return array|null Konfigurace upravené routy nebo null.
	 */
	protected function _getRouteWithoutRegions(ListingFilter $filter, $defaultRouteName)
	{
		$changed = false;

		$addressFilters = $filter->getAddressFilters();
		foreach ($addressFilters as $index => $addressFilter) {
			if ($addressFilter->region_codes) {
				$changed = true;
				$addressFilters[$index] = new AddressFilter($addressFilter->county_code, array());
			}
		}

		if (!$changed) {
			return null;
		}

		return $this->_getRouteConfigFromFilter(
			new ListingFilter($filter->toArray(), $addressFilters), $defaultRouteName
		);
	}

	/**
	 * Ostraní z filtru předané položky a vrátí konfiguraci upravené routy.
	 *
	 * Pokud nebyla vyplněna ani jedna z položek, vrací null.
	 *
	 * @param ListingFilter $filter           Vyplněný filtr nabídek.
	 * @param string        $defaultRouteName Routa na defaultní (ne-SEO) výpis
	 * @param array         $fieldsToRemove   Název polí (vlastností filtru) k odebrání.
	 *
	 * @return array|null Konfigurace upravené routy nebo null.
	 */
	protected function _getRouteWithoutFields(ListingFilter $filter, $defaultRouteName, array $fieldsToRemove)
	{
		$changed = false;

		$filterData = $filter->toArray();
		foreach ($fieldsToRemove as $fieldName) {
			if (!empty($filterData[$fieldName])) {
				$changed = true;
				unset($filterData[$fieldName]);
			}
		}

		if (!$changed) {
			return null;
		}

		return $this->_getRouteConfigFromFilter(
			new ListingFilter($filterData, $filter->getAddressFilters()), $defaultRouteName
		);
	}

	/**
	 * Převede filtr na routu s parametry.
	 *
	 * @param ListingFilter $filter           Vyplněný filtr nabídek.
	 * @param string        $defaultRouteName Routa na defaultní (ne-SEO) výpis
	 *
	 * @return array Konfigurace routy, určené filtrem (pole s klíči name, parameters).
	 */
	protected function _getRouteConfigFromFilter(ListingFilter $filter, $defaultRouteName)
	{
		$seoRouteConfig = $this->_seoRouteMatcher->getRouteConfig($filter);
		if (!empty($seoRouteConfig)) {
			return $seoRouteConfig;
		}

		return array(
			'name' => $defaultRouteName,
			'parameters' => $this->_filterConverter->getRouteParamsFromFilter($filter)
		);
	}
}
