<?php
namespace Dalten\WebBundle\Processor\TitleGenerator\ListingList;

use Dalten\WebBundle\Filter\ListingFilter;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Generátor na obecné titulky k výpisům nemovitostí.
 */
class Base implements GeneratorInterface
{
	/**
	 * Číselníky.
	 *
	 * Asociativní pole ve formátu array([název číselníku] => array([hodnota] => [label], ...), ...).
	 *
	 * @var array
	 */
	private $_codebooks;

	/**
	 * Generátor části titulků pro adresy.
	 *
	 * @var Address
	 */
	private $_addressTitleGenerator;

	/**
	 * Instance překladače.
	 *
	 * @var \Symfony\Component\Translation\TranslatorInterface
	 */
	protected $_translator;

	/**
	 * Pole zvláštních titulků - nebudou tvořeny běžným způsobem.
	 *
	 * Pole je tvořeno poli s klíči requirements (co má být ve filtru) a title_parts (části titulku,
	 * který má být vrácen).
	 *
	 * @var array
	 */
	protected $_specialTitles = array();

	/**
	 * Pole id (hodnot číselníku) typů nemovitostí, které jsou zahrnuty pod "ostatní nemovitosti".
	 *
	 * @var array
	 */
	private $_otherListingTypeIds;

	/**
	 * Nastavuje překladač a různé číselníky.
	 *
	 * $codebooks je asociativní pole ve formátu array([název číselníku] => array([hodnota] => [label], ...), ...).
	 * $specialTitles je pole ve formátu array(array('requirements' => array(...jako filtr...), 'title_parts' =>
	 *      array('advert_type' => 'prodej').
	 *
	 * @param TranslatorInterface $translator            Instance překladače.
	 * @param GeneratorInterface  $addressTitleGenerator Generátor části titulku na adresu.
	 * @param array               $codebooks             Číselníky (popis viz popis metody).
	 * @param array               $specialTitles         Pole konfigurace speciálních titulků (viz výše).
	 * @param array               $otherListingTypeIds   Pole id typů nemovitostí, které jsou zahrnuty pod "ostatní".
	 */
	public function __construct(TranslatorInterface $translator, GeneratorInterface $addressTitleGenerator,
		array $codebooks, array $specialTitles = array(), array $otherListingTypeIds = array())
	{
		$this->_translator = $translator;
		$this->_codebooks = $codebooks;
		$this->_addressTitleGenerator = $addressTitleGenerator;
		$this->_specialTitles = $specialTitles;
		$this->_otherListingTypeIds = $otherListingTypeIds;
	}

	/**
	 * Vytvoří z předaného filtru titulek pro výpis.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return string Vytvořený titulek pro filtr.
	 */
	public function createTitle(ListingFilter $filter)
	{
		$titleParts = $this->_findSpecialTitleForFilter($filter);

		if (!isset($titleParts['advert_type'])) {
			$titleParts['advert_type'] = $this->_getAdvertTypeTitlePart($filter);
		}

		if (!isset($titleParts['listing_type'])) {
			$titleParts['listing_type'] = $this->_getListingTypePart($filter);
		}

		if (!isset($titleParts['address'])) {
			$titleParts['address'] = $this->_getAddressPart($filter);
		}
		$title = implode(' ', array_filter($titleParts));

		return mb_strtoupper(mb_substr($title, 0, 1)) . mb_substr($title, 1);
	}

	/**
	 * Vrátí název typu zakázky. Pokud je vybrán víc než jeden typ, nevrátí nic.
	 *
	 * @param ListingFilter $filter Vyplněý filtr nabídek.
	 *
	 * @return string Název typu zakázky.
	 */
	protected function _getAdvertTypeTitlePart(ListingFilter $filter)
	{
		if ($filter->advert_type === array(1)) {
			return $this->_translator->trans('prodej');
		} elseif ($filter->advert_type === array(2)) {
			return $this->_translator->trans('pronájem');
		}

		return '';
	}

	/**
	 * Vrátí název typů nabídky.
	 *
	 * Pokud není žádný typ nabídky, ale je v číselnících klíč 0, vrátí místo prázdného stringu hodnotu tohoto klíče.
	 *
	 * @param \Dalten\WebBundle\Filter\ListingFilter $listingFilter Vyplněný filtr nabídek.
	 *
	 * @return string Název typů nabídky.
	 */
	protected function _getListingTypePart(ListingFilter $listingFilter)
	{
		$types = array_filter(array(9 => $this->_getCommercialListingTypeName($listingFilter)));
		// typy 2, 7, 8, 9 jsou zpracovávány jako komerční a tudíž ne tady
		if (in_array(1, $listingFilter->listing_type)) {
			$types[1] = $this->_getTranslatedTypeName(1, $listingFilter->advert_type);
		}
		if (in_array(3, $listingFilter->listing_type)) {
			$title = $this->_getTranslatedTypeName(3, $listingFilter->advert_type);
			$subTypes = $this->_getTranslatedCodebookValues($listingFilter->estate_kind, 'estate_kind');
			$types[3] = $title . ($subTypes ? ' (' . implode(', ', $subTypes) . ')' : '');
		}
		if (in_array(4, $listingFilter->listing_type)) {
			$title = $this->_getTranslatedTypeName(4, $listingFilter->advert_type);
			$subTypes = $this->_getTranslatedCodebookValues($listingFilter->flat_kind, 'flat_kind');
			$types[4] = $title . ($subTypes ? ' (' . implode(', ', $subTypes) . ')' : '');
		}
		if (in_array(5, $listingFilter->listing_type)) {
			$types[5] = $this->_getTranslatedTypeName(5, $listingFilter->advert_type);
		}
		if (in_array(6, $listingFilter->listing_type)) {
			$title = $this->_getTranslatedTypeName(6, $listingFilter->advert_type);
			$subTypes = $this->_getTranslatedCodebookValues($listingFilter->object_kind_houses, 'object_kind');
			$types[6] = $title . ($subTypes ? ' (' . implode(', ', $subTypes) . ')' : '');
		}
		if (in_array(10, $listingFilter->listing_type)) {
			$types[10] = $this->_getTranslatedTypeName(10, $listingFilter->advert_type);
		}
		if (in_array(11, $listingFilter->listing_type)) {
			$types[11] = $this->_getTranslatedTypeName(11, $listingFilter->advert_type);
		}

		if (!$listingFilter->listing_type) {
			$types[0] = $this->_getTranslatedTypeName(0, $listingFilter->advert_type);
		}

		if (count($types) == count($this->_otherListingTypeIds)
			&& !array_diff(array_keys($types), $this->_otherListingTypeIds)) {
			// tyhle typy znamenají, že je aktivní slug ostatni a tudíž zobrazíme titulek pro ostatní nemovitost
			$types = array(92 => $this->_getTranslatedTypeName(92, $listingFilter->advert_type));
		}

		return implode(', ', array_filter($types));
	}

	/**
	 * Vrátí název typů nabídky pro komerční nemovitosti.
	 *
	 * Komerční nemovitosti jsou složené z více typů a podtypů a proto je třeba je řešit odděleně.
	 *
	 * @param \Dalten\WebBundle\Filter\ListingFilter $listingFilter Vyplněný filtr nabídek.
	 *
	 * @return string Název typů nabídky (z komerčních nemovitostí).
	 */
	protected function _getCommercialListingTypeName(ListingFilter $listingFilter)
	{
		if (in_array(7, $listingFilter->listing_type) || in_array(9, $listingFilter->listing_type)
			|| in_array(2, $listingFilter->listing_type) || in_array(8, $listingFilter->listing_type)) {
			$subTypes = array();
			$selectedAdvertTypes = $listingFilter->advert_type;

			if (in_array(9, $listingFilter->listing_type)) {
				$subTypes = array_merge(
					$subTypes,
					$this->_getTranslatedCommercialSubtypeNames(
						'office_kind', $listingFilter->office_kind, $selectedAdvertTypes
					)
				);
			}

			if (in_array(7, $listingFilter->listing_type)) {
				$subTypes = array_merge(
					$subTypes,
					$this->_getTranslatedCommercialSubtypeNames(
						'hotel_kind', $listingFilter->hotel_kind, $selectedAdvertTypes
					)
				);
			}

			if (in_array(8, $listingFilter->listing_type)
				&& (!in_array(2, $listingFilter->listing_type) || !in_array(7, $listingFilter->listing_type)
					|| !in_array(9, $listingFilter->listing_type))
			) {
				// nájmení domy tam dáme pouze pokud nemáme vybrané všechny čtyři typy komerčních nemovitostí
				$subTypes[] = $this->_getTranslatedTypeName(8, $selectedAdvertTypes);
			}

			if ($subTypes) {
				return implode(', ', $subTypes);
			}

			return $this->_getTranslatedTypeName(2, $selectedAdvertTypes);
		}

		return '';
	}

	/**
	 * Vrátí přeložné (pomocí translátoru z konstruktoru) hodnoty položek číselníku.
	 *
	 * @param array  $values       Pole vybraných hodnot.
	 * @param string $codebookName Název číselníku.
	 *
	 * @return array Pole názvů vybraných hodnot. Klíčem jsou hodnoty z argumentu a hodnoty jsou labely.
	 */
	protected function _getTranslatedCodebookValues(array $values, $codebookName)
	{
		$codebook = isset($this->_codebooks[$codebookName]) ? $this->_codebooks[$codebookName] : array();
		if (empty($codebook)) {
			return array();
		}
		$return = array();
		foreach ($values as $value) {
			if (isset($codebook[$value])) {
				$return[$value] = $this->_translator->trans($codebook[$value]);
			}
		}

		return $return;
	}

	/**
	 * Vrátí přeložený a správně vyskloňovaný název typu nemovitosti.
	 *
	 * @param int   $listingType         Typ nemovitosti.
	 * @param array $selectedAdvertTypes Pole vybraných typů zakázky (prodej/pronájem).
	 *
	 * @return string|null Přeložený a správně vyskloňovaný název typu nemovitosti nebo null.
	 */
	protected function _getTranslatedTypeName($listingType, array $selectedAdvertTypes)
	{
		$codebookNameSuffix = $this->_getCodebookSuffix($selectedAdvertTypes);

		return current($this->_getTranslatedCodebookValues(array($listingType), 'listing_type_' . $codebookNameSuffix));
	}

	/**
	 * Vrátí přeložený a správně vyskloňovaný název podtypu komerční nemovitosti.
	 *
	 * @param string $subtypeName         Název podtypu (základu názvu číselníku).
	 * @param array  $values              Pole vybraných hodnot podtypu.
	 * @param array  $selectedAdvertTypes Pole vybraných typů zakázky (prodej/pronájem).
	 *
	 * @return string|null Přeložený a správně vyskloňovaný název podtypu komerční nemovitosti nebo null.
	 */
	protected function _getTranslatedCommercialSubtypeNames($subtypeName, array $values, array $selectedAdvertTypes)
	{
		$codebookNameSuffix = $this->_getCodebookSuffix($selectedAdvertTypes);

		return $this->_getTranslatedCodebookValues($values, $subtypeName . '_' . $codebookNameSuffix);
	}

	/**
	 * Vrátí část titulku, která se zabývá adresou.
	 *
	 * @param ListingFilter $filter Vyplněný filtr.
	 *
	 * @return string Část titulku, která se zabývá adresou.
	 */
	protected function _getAddressPart(ListingFilter $filter)
	{
		return $this->_addressTitleGenerator->createTitle($filter);
	}



	/**
	 * Pokusí se pro filtr najít speciální titulek z konfigurace. Pokud jej nalezne, vrátí jeho title_parts,
	 * jinak vrátí prázdné pole.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return array Nalezené části titulku nebo prázdné pole.
	 */
	protected function _findSpecialTitleForFilter(ListingFilter $filter)
	{
		foreach ($this->_specialTitles as $titleConfig) {
			$matched = true;
			foreach ($titleConfig['requirements'] as $propertyName => $expectedValue) {
				$filterValue = $filter->$propertyName;
				if (is_array($filterValue)) {
					sort($filterValue);
					$expectedValue = (array) $expectedValue;
					sort($expectedValue);
				}

				if ($expectedValue != $filterValue) {
					$matched = false;
					// skočíme hned na další config
					break;
				}
			}

			if ($matched) {
				return array_map(array($this->_translator, 'trans'), $titleConfig['title_parts']);
			}
		}

		return array();
	}

	/**
	 * Vrátí suffix k názvu codebook (default, sale, lease) na základě předaných vybraných typů zakázky.
	 *
	 * @param array $selectedAdvertTypes Pole vybraných typů zakázky.
	 *
	 * @return string Suffix číselníku (default, sale, lease).
	 */
	protected function _getCodebookSuffix(array $selectedAdvertTypes)
	{
		$codebookNameSuffix = 'default';
		if ($selectedAdvertTypes == array(1)) {
			$codebookNameSuffix = 'sale';

			return $codebookNameSuffix;
		} elseif ($selectedAdvertTypes == array(2)) {
			$codebookNameSuffix = 'lease';

			return $codebookNameSuffix;
		}

		return $codebookNameSuffix;
	}
}
