<?php
namespace Dalten\WebBundle\Twig\Extensions\Extension;

use Symfony\Component\HttpFoundation\Request;

/**
 * Obsahuje funkce na správu proměnných v http query pro stránkování.
 */
class Pagination extends \Twig_Extension
{
	/**
	 * Vrátí seznam funkcí které dokáže obsluhovat.
	 *
	 * @return array
	 */
	public function getFunctions()
	{
		return array(
			'pagination_next_page' => new \Twig_Function_Method($this, 'increasePageNumber'),
			'pagination_prev_page' => new \Twig_Function_Method($this, 'decreasePageNumber'),
			'pagination_custom_page' => new \Twig_Function_Method($this, 'customPageNumber'),
			'pagination_items_on_page' => new \Twig_Function_Method($this, 'recordsOnPageCount')
		);
	}

	/**
	 * Pokusí se navýšit proměnnou s číslem stránky v HTTP query o předaný počet.
	 *
	 * Pokud proměnná v HTTP query není přiřadí ji hodnotu 2.
	 *
	 * @param Request $request          Instance Requestu.
	 * @param integer $increment        Počet o kolik chceme navýšit proměnnou.
	 * @param string  $pageVariableName Název proměnné s číslem stránky v HTTP query.
	 *
	 * @return string Query string z requestu s upravenou proměnnou.
	 */
	public function increasePageNumber(Request $request, $increment, $pageVariableName = 'page')
	{
		$page = $request->query->get($pageVariableName, null);
		$query = $this->_getQueryParams($request);

		if ($page > 1) {
			$query[$pageVariableName] += (integer) $increment;
		} else {
			$query[$pageVariableName] = 2;
		}

		return \http_build_query($query);
	}

	/**
	 * Pokusí se snížit hodnotu proměnné s číslem stránky v HTTP query o předaný počet.
	 *
	 * Pokud proměnná v HTTP query není, nebo je menší než 2, nenastaví ji.
	 *
	 * @param Request $request          Instance Requestu.
	 * @param integer $decrement        Počet o kolik chceme ponížit proměnnou.
	 * @param string  $pageVariableName Název proměnné s číslem stránky v HTTP query.
	 *
	 * @return string Query string z requestu s upravenou proměnnou.
	 */
	public function decreasePageNumber(Request $request, $decrement, $pageVariableName = 'page')
	{
		$page = $request->query->get($pageVariableName, null);
		$query = $this->_getQueryParams($request);

		if ($page > 2) {
			$query[$pageVariableName] -= (integer) $decrement;
		} elseif (isset($query[$pageVariableName])) {
			unset($query[$pageVariableName]);
		}

		return \http_build_query($query);
	}

	/**
	 * Nastaví předanou hodnotu jako číslo stránky a vrátí Query string.
	 *
	 * @param Request $request          Instance Requestu.
	 * @param mixed   $pageNumber       Požadované číslo stránky.
	 * @param string  $pageVariableName Název proměnné s číslem stránky.
	 *
	 * @return string Query string z requestu s nastavenou proměnnou.
	 */
	public function customPageNumber(Request $request, $pageNumber, $pageVariableName = 'page')
	{
		$query = $this->_getQueryParams($request);
		$pageVariableName = (string) $pageVariableName;
		if ($pageNumber <= 1) {
			unset($query[$pageVariableName]);
		} else {
			$query[$pageVariableName] = $pageNumber;
		}

		return \http_build_query($query);
	}

	/**
	 * Nastaví počet položek na stránce na předanou hodnotu, nastaví první stranu a vrátí Query string.
	 *
	 * @param Request $request                   Instance Requestu.
	 * @param mixed   $recordsOnPage             Požadovaný počet položek na stránku.
	 * @param string  $recordsOnPageVariableName Název proměnné s počtem položek na stránku.
	 * @param string  $pageVariableName          Název proměnné s číslem stránky.
	 *
	 * @return string Query string z requestu s nastavenou proměnnou.
	 */
	public function recordsOnPageCount(Request $request, $recordsOnPage, $recordsOnPageVariableName = 'records_on_page',
		$pageVariableName = 'page')
	{
		$query = $this->_getQueryParams($request);
		$query[(string) $recordsOnPageVariableName] = $recordsOnPage;
		unset($query[(string) $pageVariableName]);

		return \http_build_query($query);
	}

	/**
	 * Vrátí jméno rozšíření.
	 *
	 * @return string
	 */
	function getName()
	{
		return 'pagination';
	}

	/**
	 * Vrátí parametry z query string jako pole.
	 *
	 * @param Request $request Instance requestu.
	 *
	 * @return array Parametry z query string jako pole.
	 */
	protected function _getQueryParams(Request $request)
	{
		return $request->query->all();
	}
}
