<?php
/**
 * Created by PhpStorm.
 * User: Karel
 * Date: 19.12.13
 * Time: 14:45
 */

namespace Dalten\WebBundle\Tests\Processor\TitleGenerator\ListingList;

use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Processor\TitleGenerator\ListingList\Base;


class BaseTest extends \PHPUnit_Framework_TestCase
{
	private $_codebooks = array(
		'listing_type_default' => array(
			2 => 'komerční nemovitosti',
			4 => 'byty',
			6 => 'domy',
			8 => 'nájemní domy',
			9 => 'komerční prostory',
			'project' => 'bagr'
		),
		'listing_type_sale' => array(
			0 => 'nemovitostí',
			2 => 'komerčních nemovitostí',
			4 => 'bytů',
			6 => 'domů',
			8 => 'nájemních domů',
			9 => 'komerčních prostor',
			92 => 'ostatních nem.',
			'project' => 'bagr'
		),
		'listing_type_lease' => array(
			2 => 'komerční nemovitosti',
			4 => 'bytu',
			6 => 'domu',
			8 => 'nájemního domu',
			9 => 'komerčního prostoru',
			'project' => 'bagr'
		),
		'flat_kind' => array(
			1 => 'garsoniéra',
			3 => '1+kk'
		),
		'office_kind_default' => array(
			1 => 'kanceláře'
		),
		'office_kind_sale' => array(
			1 => 'kanceláří'
		),
		'office_kind_lease' => array(
			1 => 'kanceláře'
		),
	);
	/** @var  \Symfony\Component\Translation\TranslatorInterface */
	private $_translator;

	protected function setUp()
	{
		$this->_translator = $this->getMockBuilder('Symfony\Component\Translation\TranslatorInterface')
			->getMock();
		$this->_translator->expects($this->any())->method('trans')->will($this->returnArgument(0));
	}

	public function testGeneratorCapitalizesFirstLetter()
	{
		$filter = new ListingFilter(array('listing_type' => 4));

		$this->assertEquals('Byty', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorReturnsCodebookValuesOfZeroIfNoListingTypeIsSet()
	{
		$filter = new ListingFilter(array('advert_type' => 1));

		$this->assertEquals('Prodej nemovitostí', $this->_getTitleForFilter($filter));
	}

	public function testBaseGeneratorDoesNotHaveSpecialHandlingForProjects()
	{
		$filter = new ListingFilter(array('only_projects' => true, 'listing_type' => 4));

		$this->assertEquals('Byty', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorGroupsSubtypesAfterTypes()
	{
		$filter = new ListingFilter(
			array('listing_type' => array(4, 6), 'flat_kind' => array(1,3), 'advert_type' => 1)
		);

		$this->assertEquals('Prodej bytů (garsoniéra, 1+kk), domů', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorGroupsSpecifiedListingTypesIntoOtherGroup()
	{
		$filter = new ListingFilter(
			array('listing_type' => array(1, 5, 10, 11), 'advert_type' => 1)
		);

		$this->assertEquals('Prodej ostatních nem.', $this->_getTitleForFilter($filter, [1, 5, 10, 11]));


		$filter = new ListingFilter(
			array('listing_type' => array(1, 10, 11), 'advert_type' => 1)
		);

		$this->assertNotEquals('Prodej ostatních nem.', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorCanUseSubtypesForCommercialListings()
	{
		$filter = new ListingFilter(
			array('listing_type' => array(8, 9), 'advert_type' => 1, 'office_kind' => 1)
		);

		$this->assertEquals('Prodej kanceláří, nájemních domů', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorShowsLabelForAppartmentBuildingsOnlyIfNotAllCommercialSubtypesAreSet()
	{
		/*
			nájemní domy jsou typ, kdežto ostatní komerční nemovitosti jsou podtypy
			proto k zobrazení titulku komerčních nemovitostí musíme většinou zvolit jeden podtyp (kanceláře),
			ale nájemní domy jsou vybrány vždy (pokud je explicitně neodškrtneme).
			Pokud ale chceme titulek pro všechny komerční nemovitosti (bez členění), tak nesmíme
			považovat nájemní domy za vybrané.

			Dříve se při zaškrtnutí typu nemovitosti "komerční objekty" objevil titulek
			"komerční nemovitosti - nájemní domy", správně má být "komerční nemovitosti".
		*/
		$filter = new ListingFilter(
			array('listing_type' => array(2, 7, 8, 9), 'advert_type' => 1)
		);

		$this->assertEquals('Prodej komerčních nemovitostí', $this->_getTitleForFilter($filter));

		$filter = new ListingFilter(
			array('listing_type' => array(8), 'advert_type' => 1)
		);

		$this->assertEquals('Prodej nájemních domů', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorShowsLabelForAppBuildingsOnlyIfNotAllCommercialSubtypesAreSetRegardlessOfOtherTypes()
	{
		/*
			Rozšiřuje výše uvedený test o situce, kdy jsou vybrány nejen všechny kom. nemovitosti, ale i nějaký
			jiný typ.

			Dříve se při zaškrtnutí typu nemovitosti "komerční objekty" a "byty" objevil titulek
			"nájemní domy, byty", správně má být "komerční nemovitosti, byty".
		*/
		$filter = new ListingFilter(
			array('listing_type' => array(2, 4, 7, 8, 9), 'advert_type' => 1)
		);

		$this->assertEquals('Prodej komerčních nemovitostí, bytů', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorGetsAddressPartFromPassedInAddressGenerator()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1),
			array(new AddressFilter(7, array()), new AddressFilter(9, array()))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)
			->will($this->returnValue('test'));

		$generator = new Base($this->_translator, $addressGenerator, $this->_codebooks);

		$this->assertEquals('Prodej bytů test', $generator->createTitle($filter));
	}

	public function testGeneratorCorrectlyChoosesProperCaseForDifferentAdvertTypes()
	{
		$filter = new ListingFilter(array('listing_type' => 4, 'advert_type' => 1));
		$this->assertEquals('Prodej bytů', $this->_getTitleForFilter($filter));

		$filter = new ListingFilter(array('listing_type' => 4));
		$this->assertEquals('Byty', $this->_getTitleForFilter($filter));

		$filter = new ListingFilter(array('listing_type' => 4, 'advert_type' => 2));
		$this->assertEquals('Pronájem bytu', $this->_getTitleForFilter($filter));
	}

	public function testGeneratorCanUsePresetSpecialTitle()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1, 'only_projects' => true)
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new Base($this->_translator, $addressGenerator, $this->_codebooks, $specialTitles);

		$this->assertEquals('Bagr jede', $generator->createTitle($filter));
	}

	public function testGeneratorTranslatesPresetSpecialTitleParts()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1, 'only_projects' => true)
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$translator = $this->getMockBuilder('Symfony\Component\Translation\TranslatorInterface')
			->getMock();
		$translator->expects($this->any())->method('trans')
			->will($this->returnCallback(function($string) { return '__' . $string; }));

		$generator = new Base($translator, $addressGenerator, $this->_codebooks, $specialTitles);

		$this->assertEquals('__Bagr __jede', $generator->createTitle($filter));
	}

	/**
	 * Bugfix test
	 *
	 * Generátor prohledával jen podle první vlastnosti, ne podle všech
	 */
	public function testGeneratorSearchesDeepForSpecialTitle()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1, 'only_projects' => true)
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4), 'advert_type' => array()),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'nejede')
			),
			array(
				'requirements' => array('listing_type' => array(4), 'advert_type' => array(1)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede')
			)
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new Base($this->_translator, $addressGenerator, $this->_codebooks, $specialTitles);

		$this->assertEquals('Bagr jede', $generator->createTitle($filter));
	}

	public function testGeneratorReturnsNormalTitleIfNoSpecialTitleIsMatched()
	{
		$filter = new ListingFilter(
			array('listing_type' => 6, 'advert_type' => 1, 'only_projects' => true)
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new Base($this->_translator, $addressGenerator, $this->_codebooks, $specialTitles);

		$this->assertEquals('Prodej domů', $generator->createTitle($filter));
	}

	public function testGeneratorCanFillMissingPartsOfSpecialTitle()
	{
		$filter = new ListingFilter(
			array('listing_type' => 4, 'advert_type' => 1, 'only_projects' => true),
			array(new AddressFilter(9, array()))
		);

		$specialTitles = array(
			array(
				'requirements' => array('listing_type' => array(4)),
				'title_parts' => array('listing_type' => 'Bagr', 'advert_type' => 'jede'))
		);

		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)
			->will($this->returnValue('kolobezka'));

		$generator = new Base($this->_translator, $addressGenerator, $this->_codebooks, $specialTitles);

		$this->assertEquals('Bagr jede kolobezka', $generator->createTitle($filter));
	}

	private function _getTitleForFilter(ListingFilter $filter, $otherListingTypeIds = [])
	{
		$addressGenerator = $this->getMockBuilder('Dalten\WebBundle\Processor\TitleGenerator\ListingList\Address')
			->disableOriginalConstructor()
			->getMock();

		$addressGenerator->expects($this->once())->method('createTitle')->with($filter)->will($this->returnValue(''));

		$generator = new Base($this->_translator, $addressGenerator, $this->_codebooks, [], $otherListingTypeIds);

		return $generator->createTitle($filter);
	}

}
