<?php

namespace Dalten\WebBundle\Breadcrumbs\Listing\Builder;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Entity\Listing;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\BreadcrumbInterface;

/**
 * Drobeček na typ nemovitosti.
 */
class ListingType implements BuilderInterface
{
	/**
	 * Konfigurace.
	 *
	 * @var array
	 */
	private $_config;

	/**
	 * Nastavuje konfiguraci.
	 *
	 * Konfigurační pole by mělo obsahovat klíče routeName a valueMap, kde valueMap je pole
	 * s klíči = id typu nemovitosti a hodnoty jsou pole s klíči label a routeParams.
	 *
	 * @param array $config Pole konfigurace. Mělo by obsahovat parametry pro všechny typy nemovitostí.
	 */
	public function __construct(array $config)
	{
		$this->_config = $config;
	}

	/**
	 * Vytvoří drobka z vyplněného modelu nabídky.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param Listing $listing Model nabídky.
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromListing(Listing $listing)
	{
		return $this->_getBreadcrumb(array($listing->listing_type));
	}

	/**
	 * Vytvoří drobka z vyplněného filtru nabídek.
	 *
	 * Pokud se pro dané údaje nepodaří drobek vytvořit, vrátí null.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return BreadcrumbInterface|null Drobek nebo null.
	 */
	public function getBreadcrumbFromFilter(ListingFilter $filter)
	{
		$filterData = $filter->toArray();
		return $this->_getBreadcrumb(
			empty($filterData['listing_type']) ? array() : (array) $filterData['listing_type']
		);
	}

	/**
	 * Vytvoří model drobečku dle předaného typu nemovitosti a vrátí jej.
	 *
	 * @param array $listingTypes Id typu nemovitosti.
	 *
	 * @return null|BreadcrumbInterface Drobek nebo null pokud jej nelze vytvořit.
	 */
	private function _getBreadcrumb(array $listingTypes)
	{
		$currentTypeConfig = $fallbackConfig = array();
		sort($listingTypes);

		foreach ($this->_config['valueMap'] as $typeConfig) {
			$acceptedValues = $typeConfig['acceptedValues'];
			sort($acceptedValues);
			if ($listingTypes === $acceptedValues) {
				$currentTypeConfig = $typeConfig;
				break;
			}
			$listingType = current($listingTypes);
			if (in_array($listingType, $typeConfig['acceptedValues'])) {
				$fallbackConfig = $typeConfig;
			}
		}

		if (empty($currentTypeConfig) && !empty($fallbackConfig)) {
			$currentTypeConfig = $fallbackConfig;
		}

		if (empty($currentTypeConfig)) {
			return null;
		}

		return new Breadcrumb(
			$currentTypeConfig['label'], $this->_config['routeName'], $currentTypeConfig['routeParams']
		);
	}
}
