<?php
namespace Dalten\WebBundle\Processor\ListingListData;

use Dalten\WebBundle\Filter\ListingFilter;

/**
 * Jednoduchý immutable value-object, který uchovává parametry nutné pro vytvoření výpisu nabídek.
 */
class Data
{
	/**
	 * Jedná se o výpis zlevněných nemovitostí?
	 *
	 * @var bool
	 */
	private $_discounted = false;

	/**
	 * Jedná se o výpis zahraničních nemovitostí?
	 *
	 * @var bool
	 */
	private $_foreign = false;

	/**
	 * Jedná se o výpis oblíbených nemovitostí?
	 *
	 * @var bool
	 */
	private $_favorite = false;

	/**
	 * Jedná se o výpis nemovitostí s dnem otevřených dveří?
	 *
	 * @var bool
	 */
	private $_openDay = false;

	/**
	 * Instance filtru.
	 *
	 * @var ListingFilter
	 */
	private $_filter;

	/**
	 * Pole, podle kterého řadíme výsledky.
	 *
	 * @var string
	 */
	private $_orderByField;

	/**
	 * Řadíme vzestupně?
	 *
	 * @var bool
	 */
	private $_orderByAscending;

	/**
	 * Pole id oblíbených nabídek.
	 *
	 * @var int[]
	 */
	private $_favoriteListingIds = array();

	/**
	 * Číslo stránky.
	 *
	 * @var int
	 */
	private $_currentPageNumber;

	/**
	 * Řazení dle data vložení nabídky.
	 */
	const ORDER_BY_PUBLISHED_DATE = 'order-by-published-date';

	/**
	 * Řazení dle ceny.
	 */
	const ORDER_BY_PRICE = 'order-by-price';

	/**
	 * Nastavení vlastností objektu.
	 *
	 * @param ListingFilter $filter             Filtr nabídek.
	 * @param bool          $isDiscounted       Jedná se o výpis zlevněných nabídek?
	 * @param bool          $isForeign          Jedná se o výpis zahraničních nabídek?
	 * @param bool          $isFavorite         Jedná se o výpis oblíbených nabídek?
	 * @param bool          $isOpenDay          Jedná se o výpis nabídek se dnem otevřených dveří?
	 * @param int[]         $favoriteListingIds Pole ID oblíbených nabídek.
	 * @param int           $currentPageNumber  Číslo aktuální stránky (1+).
	 * @param string        $orderByField       Pole, dle kterého řadíme (konstanta třídy ORDER_BY_*).
	 * @param bool          $orderByAscending   Řadíme vzestupně?
	 */
	public function __construct(ListingFilter $filter, $isDiscounted, $isForeign, $isFavorite, $isOpenDay,
		array $favoriteListingIds, $currentPageNumber, $orderByField, $orderByAscending)
	{
		$this->_discounted = (bool) $isDiscounted;
		$this->_foreign = (bool) $isForeign;
		$this->_favorite = (bool) $isFavorite;
		$this->_openDay = (bool) $isOpenDay;
		$this->_filter = $filter;
		$orderByField = (string) $orderByField;
		if (!in_array($orderByField, array(self::ORDER_BY_PUBLISHED_DATE, self::ORDER_BY_PRICE))) {
			throw new \InvalidArgumentException(
				sprintf('Řazení "%s" není platné. Povoleny jsou pouze konstanty této třídy ORDER_BY_*.', $orderByField)
			);
		}
		$this->_orderByField = $orderByField;
		$this->_orderByAscending = (bool) $orderByAscending;
		$this->_currentPageNumber = (int) $currentPageNumber;
		$this->_favoriteListingIds = $favoriteListingIds;
	}

	/**
	 * Jedná se o výpis zlevněných nabídek?
	 *
	 * @return boolean Jedná se o výpis zlevněných nabídek?
	 */
	public function isDiscounted()
	{
		return $this->_discounted;
	}

	/**
	 * Jedná se o výpis oblíbených nabídek?
	 *
	 * @return boolean Jedná se o výpis oblíbených nabídek?
	 */
	public function isFavorite()
	{
		return $this->_favorite;
	}

	/**
	 * Vrací vloženou instanci filtru.
	 *
	 * @return ListingFilter Vrací vloženou instanci filtru.
	 */
	public function getFilter()
	{
		return $this->_filter;
	}

	/**
	 * Jedná se o výpis zahraničních nabídek?
	 *
	 * @return boolean Jedná se o výpis zahraničních nabídek?
	 */
	public function isForeign()
	{
		return $this->_foreign;
	}

	/**
	 * Jedná se o výpis nabídek se dnem otevřených dveří?
	 *
	 * @return boolean Jedná se o výpis nabídek se dnem otevřených dveří?
	 */
	public function isOpenDay()
	{
		return $this->_openDay;
	}

	/**
	 * Řadíme dle ceny?
	 *
	 * @return bool Řadíme dle ceny?
	 */
	public function isOrderedByPrice()
	{
		return $this->_orderByField === self::ORDER_BY_PRICE;
	}

	/**
	 * Řadíme dle data vložení?
	 *
	 * @return bool Řadíme dle data vložení?
	 */
	public function isOrderedByPublishedDate()
	{
		return $this->_orderByField === self::ORDER_BY_PUBLISHED_DATE;
	}

	/**
	 * Řadíme vzestupně?
	 *
	 * @return bool
	 */
	public function isInAscendingOrder()
	{
		return $this->_orderByAscending;
	}

	/**
	 * Vrací nastavené pole ID oblíbených nabídek.
	 *
	 * @return int[] Nastavené pole ID oblíbených nabídek.
	 */
	public function getFavoriteListingIds()
	{
		return $this->_favoriteListingIds;
	}

	/**
	 * Vrací aktuální číslo stránky.
	 *
	 * @return int Aktuální číslo stránky.
	 */
	public function getCurrentPageNumber()
	{
		return $this->_currentPageNumber;
	}

}
