<?php
namespace Dalten\WebBundle\SeoRoute\Matcher\ListingFilter;

use Dalten\Util\StringHelper;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Processor\SeoUrl\AddressSlugGenerator;

/**
 * Matcher, který převádí část adresního filtru na slug kraje.
 *
 * Interně využívá addressSlugGenerator, pouze upravuje jeho výstup dle očekávání.
 */
class LocalityFragmentMatcher implements FragmentMatcherInterface
{
	/**
	 * Vnitřní generátor slugů.
	 *
	 * @var AddressSlugGenerator
	 */
	private $_generator;

	/**
	 * Cache výsledků pro jednotlivé adresní filtry.
	 *
	 * @var array
	 */
	private $_cache = array();

	/**
	 * Nastavuje konfiguraci matcheru.
	 *
	 * @param AddressSlugGenerator $generator Generátor adresních slugů.
	 */
	public function __construct(AddressSlugGenerator $generator)
	{
		$this->_generator = $generator;
	}

	/**
	 * Vrací pole parametrů pro routu, obsahující matchnutý fragment.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return array Pole parametrů nalezené routy a nebo prázdné pole.
	 */
	public function getRouteParamsForFilter(ListingFilter $filter)
	{
		$key = $this->_computeCacheKey($filter);
		if (isset($this->_cache[$key])) {
			return $this->_cache[$key];
		}
		return $this->_cache[$key] = $this->_generator->getAddressUrlParams($filter);
	}

	/**
	 * Vrátí pole názvů vlastností filtru, které byly zpracovány tímto fragmentem pro daný filtr.
	 *
	 * Vrací prázdné pole, pokud daný filtr není pokryt.
	 *
	 * UPOZORNĚNÍ: Protože adresa není vlastností filtru, vrací tato metoda názvy parametrů routy a nikoli filtru!
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return array Pole názvů vlastností filtru, byly zpracovány tímto fragmentem pro daný filtr.
	 */
	public function getMatchedFilterParamsForFilter(ListingFilter $filter)
	{
		return array_keys(array_filter($this->getRouteParamsForFilter($filter)));
	}

	/**
	 * Vypočítá unikátní identifikátor adresní části filtru.
	 *
	 * Identifikátor bude stejný pro stejné adresní filtry (bez ohledu na pořadí).
	 *
	 * @param ListingFilter $filter
	 *
	 * @return string
	 */
	private function _computeCacheKey(ListingFilter $filter)
	{
		$counties = $regions = array();
		foreach ($filter->getAddressFilters() as $addressFilter) {
			$counties[] = $addressFilter->county_code;
			$regions = array_merge($regions, $addressFilter->region_codes);
		}

		sort($counties);
		$regions = array_unique($regions);
		sort($regions);

		return implode('-', $counties) . '|' . implode('-', $regions);
	}
}

