<?php
namespace Dalten\WebBundle\Translation\Extractor\JMSTranslationBundle;

use JMS\TranslationBundle\Model\Message;
use JMS\TranslationBundle\Translation\ExtractorInterface;
use JMS\TranslationBundle\Model\MessageCatalogue;

/**
 * Získává texty na překlad z víceroměrných polí.
 */
class RecursiveArrayValuesExtractor implements ExtractorInterface
{
	private $_domain = 'messages';

	/**
	 * Nastavení polí na překlad.
	 *
	 * Jedná se o pole, jehož prvky jsou pole s klíči data a paths.
	 * * data = pole k překladu
	 * * paths = Pole cest v poli data k samotným textům. Pokud je prázdné, bude překládáno celé pole data.
	 *
	 * Příklad:
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [] } => přeloženo "bagr" a "kuk"
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [test.ahoj] } => přeloženo pouze "bagr"
	 *
	 * @var array
	 */
	private $_itemConfig;

	/**
	 * Nastavuje konfiguraci polí na překlad.
	 *
	 * $itemConfig je pole, jehož prvky jsou pole s klíči data a paths.
	 * * data = pole k překladu
	 * * paths = Pole cest v poli data k samotným textům. Pokud je prázdné, bude překládáno celé pole data.
	 *
	 * Příklad:
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [] } => přeloženo "bagr" a "kuk"
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [test.ahoj] } => přeloženo pouze "bagr"
	 *
	 * @param array $itemConfig Pole s konfigurací - viz popis metody.
	 */
	public function __construct(array $itemConfig)
	{
		$this->_itemConfig = $itemConfig;
	}

	/**
	 * Extracts translation messages from a template directory to the catalogue.
	 *
	 * @return MessageCatalogue
	 */
	public function extract()
	{
		$catalogue = new MessageCatalogue();
		foreach ($this->_itemConfig as $config) {
			$this->_extractTermsFromArray($catalogue, $config['data'], $config['paths']);
		}

		return $catalogue;
	}

	/**
	 * Získá pole stringů na překlad ze zdrojového pole, které může filtrovat pomocí předaných cest.
	 *
	 * Popis funkcionality cest je v popisu konstruktoru.
	 *
	 * @param MessageCatalogue $catalogue Překladový katalog.
	 * @param array|string     $data      Data k překladu (obvykle pole, může být i string).
	 * @param array            $paths     Seznam cest, na kterých se v poli nachází řetězce k překladu.
	 *
	 * @return array Texty k překladu.
	 */
	private function _extractTermsFromArray(MessageCatalogue $catalogue, $data, array $paths = array())
	{
		$translations = array();
		$filteredData = $this->_filterDataByPaths($data, $paths);


		foreach ($filteredData as $item) {
			if (is_array($item)) {
				$translations = array_merge($translations, $this->_extractTermsFromArray($catalogue, $item, array()));
			} else {
				$catalogue->add(new Message($item, $this->_domain));
			}
		}

		return $translations;
	}

	/**
	 * Získá ze zdrojového pole texty na základě předaných cest.
	 *
	 * Popis funkcionality cest je v popisu konstruktoru.
	 *
	 * @param array|string $data  Data k překladu (obvykle pole, může být i string).
	 * @param array        $paths Seznam cest, na kterých se v poli nachází řetězce k překladu.
	 *
	 * @return array Texty k překladu.
	 *
	 * @throws \InvalidArgumentException Pokud se v poli nepodaří najít určenou cestu.
	 */
	private function _filterDataByPaths($data, $paths)
	{
		if (!is_array($data)) {
			$data = (array) $data;
		}

		if (empty($paths)) {
			return $data;
		}

		$filteredData = array();
		foreach ($paths as $path) {
			$item = $data;
			foreach(explode('.', $path) as $pathPart) {
				if (!isset($item[$pathPart])) {
					throw new \InvalidArgumentException(
						sprintf(
							'Nepodařilo se najít cestu "%s" (resp. její část "%s") v %s',
							$paths, $pathPart, print_r($data, true)
						)
					);
				}
				$item = $item[$pathPart];
			}
			$filteredData[] = $item;
		}

		return $filteredData;
	}
}
