<?php

namespace Dalten\Tests\Date;

use \Dalten\Date\Date;

class DateTest extends \PHPUnit_Framework_TestCase
{
	public function testCompareDate()
	{
		$today = new Date('now');

		$this->assertEquals(0, $today->compareDate(new Date('now')));
		$this->assertEquals(1, $today->compareDate(Date::now()->sub('1 day')));
		$this->assertEquals(1, $today->compareDate(null));
		$this->assertEquals(-1, $today->compareDate(Date::now()->add('3 day')));
	}

	public function testDateIsLater()
	{
		$today = new Date('now');

		$this->assertTrue($today->isDateLater(Date::now()->sub('1 day')));
		$this->assertTrue($today->isDateLater(null));
		$this->assertFalse($today->isDateLater(Date::now()->add('2 day')));
	}

	public function testDateIsEarlier()
	{
		$today = new Date('now');

		$this->assertFalse($today->isDateEarlier(Date::now()->sub('1 day')));
		$this->assertTrue($today->isDateEarlier(Date::now()->add('2 day')));
	}

	public function testIntervalCanBePassedToConstructor()
	{
		$date = new Date(sprintf('now -%d days', 7));
		$expected = Date::now()->sub('7 day')->format(Date::DATE);
		$this->assertSame($expected, $date->format(Date::DATE));
	}

	public function testIsLaterOrEqual()
	{
		$now = new Date();
		$later = Date::now()->add('7 days');

		$this->assertTrue($later->isLaterOrEqual($now));
		$this->assertTrue($now->isLaterOrEqual($now));
		$this->assertFalse($now->isLaterOrEqual($later));
	}

	public function testIsDateLaterOrEqual()
	{
		$now = new Date();
		$later = Date::now()->add('7 days');

		$this->assertTrue($later->isDateLaterOrEqual($now));
		$this->assertTrue($now->isDateLaterOrEqual($now));
		$this->assertFalse($now->isDateLaterOrEqual($later));
	}

	public function testIsEarlierOrEqual()
	{
		$now = new Date();
		$earlier = Date::now()->sub('7 days');

		$this->assertFalse($now->isEarlierOrEqual($earlier));
		$this->assertTrue($earlier->isEarlierOrEqual($now));
		$this->assertTrue($now->isEarlierOrEqual($now));
	}

	public function testIsDateEarlierOrEqual()
	{
		$now = new Date();
		$earlier = Date::now()->sub('7 days');

		$this->assertFalse($now->isDateEarlierOrEqual($earlier));
		$this->assertTrue($earlier->isDateEarlierOrEqual($now));
		$this->assertTrue($now->isDateEarlierOrEqual($now));
	}

	public function test__construct()
	{
		$now = date(DATE_ISO8601);
		$timezone = new \DateTimeZone('Europe/Prague');

		$date = new Date($now, $timezone);
		$this->assertEquals($now, $date->format(DATE_ISO8601));
	}

	public function test__constructThrowException()
	{
		$exceptionFired = false;
		try {
			new Date('kkkk');
		} catch (\Exception $e) {
			$exceptionFired = true;
		}

		$this->assertTrue($exceptionFired, 'Měla být vyhozena výjimka');
	}

	public function testAddIntervalAsString()
	{
		$now = new Date();
		$hour = $now->format('H');
		$now->add('+1 hour');
		$expectedHour = $hour + 1;
		// co kdyby to bylo přes půlnoc? :)
		$expectedHour = $expectedHour % 24;
		$this->assertEquals($expectedHour, $now->format('H'));
	}

	public function testAddIntervalAsDateIntervalObject()
	{
		$now = new Date();
		$year = $now->format('Y');

		$interval = new \DateInterval('P1Y');
		$now->add($interval);
		$this->assertEquals($year + 1, $now->format('Y'));
	}

	public function testAddTime()
	{
		$timestamp = '1.1.2011 00:00:00';
		$expectedTime = '13:33:33';
		$date = new Date($timestamp);
		$date->addTime($expectedTime);

		$this->assertEquals($expectedTime, $date->format('H:i:s'));
	}

	public function testSubstractRelativeTime()
	{
		$timestamp = '17.11.1989 18:00:00';
		$velvetDate = new Date($timestamp);
		$velvetDate->sub('18 hours');

		$this->assertEquals('00:00:00', $velvetDate->format('H:i:s'));
	}

	public function testSet()
	{
		$date = new Date();
		$this->assertEquals('1999-09-09', $date->set(1999, 9, 9)->format('Y-m-d'));
	}

	public function testCompareGreaterThan()
	{
		$date = new Date();
		$lessDate = clone $date;
		$lessDate->sub('4 hours');

		$this->assertEquals(1, $date->compare($lessDate));
	}

	public function testCompareLessThan()
	{
		$date = new Date();
		$lessDate = clone $date;
		$lessDate->add('4 hours');

		$this->assertEquals(-1, $date->compare($lessDate));
	}

	public function testCompareEquals()
	{
		$date = new Date();
		$lessDate = clone $date;

		$this->assertEquals(0, $date->compare($lessDate));
	}

	public function testIsLater()
	{
		$now = new Date();
		$this->assertTrue($now->isLater(new Date('-1 day')));
	}

	public function testIsEarlier()
	{
		$now = new Date();
		$this->assertTrue($now->isEarlier(new Date('1 day')));
	}

	public function testFormat()
	{
		$timestamp = '16.6.1966 06:06:06';
		$dateObject = new Date($timestamp);

		$unixtimestamp = \strtotime($timestamp);
		$this->assertEquals(\date(\DateTime::ISO8601, $unixtimestamp), $dateObject->format(\DateTime::ISO8601));
	}

	public function testGetTimeStamp()
	{
		$date = new Date('2009-02-14 00:31:30');
		$this->assertEquals('1234567890', $date->getTimestamp());
	}

	public function testNow()
	{
		$this->assertInstanceOf('Dalten\Date\Date', Date::now());
	}

	public function testCloneDateObject()
	{
		$object = new Date();
		$secondDate = clone $object;
		$this->assertEquals($object->getTimestamp(), $secondDate->getTimestamp());
		$this->assertNotSame($object->getDateTime(), $secondDate->getDateTime());
	}

	public function testGetDateTime()
	{
		$this->assertInstanceOf('DateTime', Date::now()->getDateTime());
	}

	public function testMagicToString()
	{
		$date = new Date();
		$this->assertEquals($date->format(Date::DATE_TIME), (string) $date);
	}

	public function testCompareAcceptsString()
	{
		$date = new Date();

		$this->assertTrue($date->isEarlier('+1 day'));
	}

	public function testCompareAcceptsDateTime()
	{
		$date = new Date();

		$this->assertTrue($date->isEarlier(new \DateTime('+1 day')));
	}

	public function testConstructorDoesNotCreateNewInstanceOfDateTimeIfPassedAnInstanceOfDateTime()
	{
		$dt = new \DateTime('last monday');
		$date = new Date($dt);

		$this->assertSame($dt, $date->getDateTime());
	}
}
