<?php
namespace Dalten\Tests\WebBundle\Twig\Extensions\Extension;

use Dalten\WebBundle\Twig\Extensions\Extension\NumberFormat;

/**
 * @group dalten
 */
class NumberFormatTest extends \PHPUnit_Framework_TestCase
{
	/**
	 * @var NumberFormat
	 */
	private $_extension;

	public function setUp()
	{
		$this->_extension = new NumberFormat();
	}

	public function testGetFilters()
	{
		$filters = $this->_extension->getFilters();
		foreach (array('number', 'currency', 'spellout', 'dechex') as $filter) {
			$this->assertArrayHasKey($filter, $filters);
			$this->assertInstanceOf('Twig_Filter_Method', $filters[$filter]);
		}
	}

	public function testGetName()
	{
		$this->assertEquals('numberFormat', $this->_extension->getName());
	}

	/**
	 * @param $value
	 * @param null $locale
	 * @param $expectedResult
	 *
	 * @dataProvider formatNumberDataProvider
	 */
	public function testFormatNumber($value, $locale = null, $expectedResult)
	{
		$result = $this->_extension->formatNumber($value, $locale);
		$this->assertEquals($expectedResult, $result);
	}

	public function formatNumberDataProvider()
	{
		return array(
			array(
				1400.7,	// $value
				'en',	// $locale
				'1,400.7' // $expectedResult
			),
			array(
				'1000',	// $value
				'en',	// $locale
				'1,000' // $expectedResult
			),
			array(
				'blah',	// $value
				'en',	// $locale
				'0' // $expectedResult
			),
			array(
				'1000',	// $value
				null,	// $locale
				'1,000' // $expectedResult
			)
		);
	}

	/**
	 * @param $value
	 * @param $expectedResult
	 *
	 * @dataProvider decHexDataProvider
	 */
	public function testFormatDecHex($value, $expectedResult)
	{
		$result = $this->_extension->formatDecHex($value);
		$this->assertEquals($expectedResult, $result);
	}

	public function decHexDataProvider()
	{
		return array(
			array(
				'11',	// $value
				'b'		// $expectedResult
			),
			array(
				'4123',	// $value
				'101b'	// $expectedResult
			),
			array(
				'abs',	// $value
				'0'	// $expectedResult
			)
		);
	}

	public function testSetCurrency()
	{
		$dataProvider = array(
			'value' => 1000,
			'currency' => 'USD',
			'expectedResult' => '$1,000.00'
		);

		$this->_extension->setCurrency($dataProvider['currency']);
		$result = $this->_extension->formatCurrency($dataProvider['value']);
		$this->assertEquals($dataProvider['expectedResult'], $result);
	}

	/**
	 * @param $value
	 * @param null $locale
	 * @param null $currency
	 * @param $expectedResult
	 *
	 * @dataProvider formatCurrencyDataProvider
	 */
	public function testFormatCurrency($value, $currency = null, $locale = null, $expectedResult)
	{
		$result = $this->_extension->formatCurrency($value, $currency, $locale);
		$this->assertEquals($expectedResult, $result);
	}

	/**
	 * @expectedException InvalidArgumentException
	 */
	public function testFormatCurrencyNoCurrencyException($value = 1000, $currency = null, $locale = 'cs')
	{
		$this->_extension->formatCurrency($value, $currency, $locale);
	}

	/**
	 * @expectedException InvalidArgumentException
	 */
	public function testFormatCurrencyCantFormatException($value = 1000, $currency = 'nesmyslBlah')
	{
		$this->_extension->formatCurrency($value, $currency);
	}

	public function formatCurrencyDataProvider()
	{
		return array(
			array(
				null,	// $value
				null,	// $currency
				null,	// $locale
				null	// $expectedResult
			),
			array(
				1000.5,	// $value
				'CZK',	// $currency
				'en',	// $locale
				'CZK1,000.50'	// $expectedResult
			),
			array(
				4721.4,	// $value
				'USD',	// $currency
				null,	// $locale
				'$4,721.40'	// $expectedResult
			)
		);
	}

	/**
	 * @param $value
	 * @param null $locale
	 * @param $expectedResult
	 *
	 * @dataProvider formatSpelledOutNumberDataProvider
	 */
	public function testFormatSpelledOutNumber($value, $locale = null, $expectedResult)
	{
		$result = $this->_extension->formatSpelledOutNumber($value, $locale);
		$this->assertEquals($expectedResult, $result);
	}

	public function formatSpelledOutNumberDataProvider()
	{
		return array(
			array(
				null,		// $value
				'en',		// $locale
				null		// $expectedResult
			),
			array(
				1057.4,		// $value
				'en',		// $locale
				'one thousand fifty-seven point four'	// $expectedResult
			),
			array(
				'954.3',	// $value
				'cs',		// $locale
				'devět set padesát čtyři čárka tři'		// $expectedResult
			),
			array(
				'blahLeh',	// $value
				'cs',		// $locale
				'nula'		// $expectedResult
			)
		);
	}
}
