<?php

namespace Dalten\WebAdminBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;

use FrontBundle\Entity\Reference;
use Dalten\WebAdminBundle\Form\ReferenceType;

/**
 * Reference controller.
 *
 */
class ReferenceController extends Controller
{
    /**
     * Lists all Reference entities.
     *
     */
    public function indexAction()
    {
        $this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

        $em = $this->getDoctrine()->getManager();

        $references = $em->getRepository('DaltenWebAdminBundle:Reference')->findAll();

        return $this->render('@DaltenWebAdmin/reference/index.html.twig', array(
            'references' => $references,
        ));
    }

    /**
     * Creates a new Reference entity.
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        $this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

        $reference = new Reference();
        $form = $this->createForm($this->getParameter('dalten_web_admin.form.reference.class'), $reference, ['translation_languages'=>$this->getParameter('dalten_web_admin.translation_languages')]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->persist($reference);
            $em->flush();

            return $this->redirectToRoute('webadmin_reference_edit', array('id' => $reference->getId()));
        }

        return $this->render('@DaltenWebAdmin/reference/new.html.twig', array(
            'reference' => $reference,
            'form' => $form->createView(),
        ));
    }

    /**
     * Finds and displays a Reference entity.
     * @param Reference $reference
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Reference $reference)
    {
        $this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

        $deleteForm = $this->createDeleteForm($reference);

        return $this->render('@DaltenWebAdmin/reference/show.html.twig', array(
            'reference' => $reference,
            'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Displays a form to edit an existing Reference entity.
     * @param Request $request
     * @param Reference $reference
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Reference $reference)
    {
        $this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

        $deleteForm = $this->createDeleteForm($reference);
        $editForm = $this->createForm($this->getParameter('dalten_web_admin.form.reference.class'), $reference, ['translation_languages'=>$this->getParameter('dalten_web_admin.translation_languages')]);
        $editForm->handleRequest($request);

        if ($editForm->isSubmitted() && $editForm->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->persist($reference);
            $em->flush();

            return $this->redirectToRoute('webadmin_reference_edit', array('id' => $reference->getId()));
        }

        return $this->render('@DaltenWebAdmin/reference/edit.html.twig', array(
            'reference' => $reference,
            'edit_form' => $editForm->createView(),
            'delete_form' => $deleteForm->createView(),
        ));
    }

    /**
     * Deletes a Reference entity.
     * @param Request $request
     * @param Reference $reference
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Request $request, Reference $reference)
    {
        $this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

        $form = $this->createDeleteForm($reference);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em = $this->getDoctrine()->getManager();
            $em->remove($reference);
            $em->flush();
        }

        return $this->redirectToRoute('webadmin_reference_index');
    }

    /**
     * Creates a form to delete a Reference entity.
     *
     * @param Reference $reference The Reference entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createDeleteForm(Reference $reference)
    {
        return $this->createFormBuilder()
            ->setAction($this->generateUrl('webadmin_reference_delete', array('id' => $reference->getId())))
            ->setMethod('DELETE')
            ->getForm()
        ;
    }
}
