<?php

namespace Dalten\WebAdminBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;

use App\FrontBundle\Entity\Article;
use App\FrontBundle\Form\ArticleType;

use Doctrine\ORM\Tools\Pagination\Paginator;

use Dalten\WebAdminBundle\Processor\ArticleStorage;

/**
 * Article controller.
 *
 */
class ArticleController extends Controller
{

    protected function _getArticleStorage()
    {
        $em = $this->getDoctrine()->getManager();
        return new ArticleStorage($em);
    }

	/**
	 * Lists all Article entities.
	 *
	 */
	public function indexAction(Request $request)
	{
		$this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

		$em = $this->getDoctrine()->getManager();

		$idSection = $request->get('id_section');

		$sections = $em->getRepository('App\FrontBundle\Entity\Section')->findBy(array(), array('name'=>'ASC'));

		$qa = $this->get('doctrine.orm.entity_manager')->createQueryBuilder();
		$qa->select('a')->from('App\FrontBundle\Entity\Article', 'a');

		if (!empty($idSection)) {
			$qa->where('a.idSection=:section');
			$qa->setParameter(':section', $idSection);
		}

		$pager = $this->_paginate($qa, 100, $request->get('page', 1));

		return $this->render('DaltenWebAdminBundle:articles:index.html.twig', array(
			'sections' => $sections,
			'articles' => $pager,
			'paging' => $pager
		));
	}

	/**
	 * Creates a new Article entity.
	 *
	 */
	public function newAction(Request $request)
	{
		$this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');
		
		/** @var \Dalten\WebAdminBundle\Processor\ArticleStorage $articleStorage */
		$articleStorage = $this->_getArticleStorage();
		$articleStorage->setUser($this->getUser());

		$article = new Article();
		$form = $this->createForm($this->getParameter('dalten_web_admin.form.article.class'), $article, ['broker_selection'=>$this->getParameter('dalten_web_admin.broker_selection')]);
		$form->handleRequest($request);

		$translationOf = $request->get('translation_of');
		$translatedLang = $request->get('lang');

		
		if ($form->isSubmitted() && $form->isValid()) {
			if ($request->get('publish')) {
				$articleStorage->publish($article);
				$this->_addTranslationLink($translationOf, $translatedLang, $article);
                return $this->redirectToRoute('webadmin_articles_edit', array('id' => $article->getId()));
			} else {
				$revNum = $articleStorage->save($article);
				$this->_addTranslationLink($translationOf, $translatedLang, $article);
				return $this->redirectToRoute('webadmin_articles_edit', array('id'=>$article->getId(), 'revnum'=>$revNum));
			}

		}

		return $this->render('DaltenWebAdminBundle:articles:new.html.twig', array(
			'article' => $article,
			'form' => $form->createView(),
		));
	}

	protected function _addTranslationLink($translationOf, $translatedLang, $article)
	{
		if ($translationOf && $translatedLang) {
			$em = $this->getDoctrine()->getManager();
			$translationLink = new \App\FrontBundle\Entity\Translation;
			$translationLink->setOriginalId((int) $translationOf);
			$translationLink->setTranslatedId((int) $article->getId());
			$translationLink->setLanguage($translatedLang);
			$em->persist($translationLink);
			$em->flush();		
		}
		
	}

	/**
	 * Finds and displays a Article entity.
	 *
	 */
	public function showAction(Article $article)
	{
		$this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

		$deleteForm = $this->createDeleteForm($article);

		return $this->render('DaltenWebAdminBundle:articles:show.html.twig', array(
			'article' => $article,
			'delete_form' => $deleteForm->createView(),
		));
	}

	/**
	 * Displays a form to edit an existing Article entity.
	 *
	 */
	public function editAction(Request $request, Article $article)
	{
		$this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

		/** @var \Dalten\WebAdminBundle\Processor\ArticleStorage $articleStorage */
		$articleStorage = $this->_getArticleStorage();
		$articleStorage->setUser($this->getUser());

		$publishedRevision = $latestRevision = $currentRevision = $article->getRevisionNumber();

		$history = $articleStorage->getHistory($article);

		if (isset($history[0])) {
			$latestRevision = $history[0]->getRevisionNumber();
		}

		if ($request->get('revnum')) {
			$article = $articleStorage->getRevision($article, $request->get('revnum'));
			$currentRevision = $article->getRevisionNumber();
		}

		$deleteForm = $this->createDeleteForm($article);
		$editForm = $this->createForm($this->getParameter('dalten_web_admin.form.article.class'), $article, ['broker_selection'=>$this->getParameter('dalten_web_admin.broker_selection')]);
		$editForm->handleRequest($request);

		if ($editForm->isSubmitted() && $editForm->isValid()) {

			if ($request->get('publish')) {
				$articleStorage->publish($article);
                return $this->redirectToRoute('webadmin_articles_edit', array('id' => $article->getId()));
			} else {
				$revNum = $articleStorage->save($article);
				return $this->redirectToRoute('webadmin_articles_edit', array('id'=>$article->getId(), 'revnum'=>$revNum));
			}
		}

		$languages = $this->getParameter('dalten_web_admin.translation_languages');
		$translations = $this->_getTranslations($article->getId());

		return $this->render('DaltenWebAdminBundle:articles:edit.html.twig', array(
			'article' => $article,
			'edit_form' => $editForm->createView(),
			'delete_form' => $deleteForm->createView(),
			'history' => $history,
			'latest_revision' => $latestRevision, // nejnovější verze
			'published_revision' => $publishedRevision, // publikovaná verze
			'current_revision' => $currentRevision, // právě upravovaná verze,
			'languages' => $languages,
			'translations' => $translations
		));
	}



	protected function _getTranslations($articleId)
	{
		$languages = $this->getParameter('dalten_web_admin.translation_languages');
		
		$em = $this->getDoctrine()->getManager();
		$repo = $em->getRepository('App\FrontBundle\Entity\Translation');
		$articleRepo = $em->getRepository('App\FrontBundle\Entity\Article');

		$originalArticleLink = $repo->findOneBy(array('translated_id'=>$articleId));
		if (!empty($originalArticleLink)) {
			$articleId = $originalArticleLink->getOriginalId();
		}

		$links = $repo->findBy(array('original_id'=>$articleId));

		if (count($links)==0) {
			return array();
		}

		$byLangs = array();
		foreach ($languages as $langCode => $langName) {
			$byLangs[$langCode]['language_code'] = $langCode;
			$byLangs[$langCode]['language_name'] = $langName;
			$byLangs[$langCode]['original_id'] = $articleId;
			$byLangs[$langCode]['translated'] = false;
		}

		foreach ($links as $link) {
			$langCode = $link->getLanguage();
			$artId = $link->getTranslatedId();

			$translatedArt = $articleRepo->findOneBy(array('id'=>$artId));

			$byLangs[$langCode]['translated'] = true;
			$byLangs[$langCode]['translated_by'] = $artId;
			$byLangs[$langCode]['translated_by'] = $artId;
			$byLangs[$langCode]['translated_name'] = $translatedArt->getName();

 		}

 		return $byLangs;
	}

	public function setLanguageAction(Request $request)
	{
		$this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

		$em = $this->getDoctrine()->getManager();
		//$repo = $em->getRepository('DaltenWebAdminBundle:Translation');

		$originalId = (int) $request->get('original_id');
		$language = $request->get('original_language');

		$trans = new \App\FrontBundle\Entity\Translation;
		$trans->setOriginalId($originalId);
		$trans->setLanguage($language);
		$trans->setTranslatedId($originalId);

		$em->persist($trans);
		$em->flush();

		return $this->redirectToRoute('webadmin_articles_edit', array('id'=>$originalId));
	}


	/**
	 * Deletes a Article entity.
	 *
	 */
	public function deleteAction(Request $request, Article $article)
	{
		$this->denyAccessUnlessGranted('ROLE_USER', null, 'Only for logged-in users.');

		$form = $this->createDeleteForm($article);
		$form->handleRequest($request);

		if ($form->isSubmitted() && $form->isValid()) {
			$em = $this->getDoctrine()->getManager();
			$em->remove($article);
			$em->flush();
		}

		return $this->redirectToRoute('webadmin_articles_index');
	}

	/**
	 * Creates a form to delete a Article entity.
	 *
	 * @param Article $article The Article entity
	 *
	 * @return \Symfony\Component\Form\Form The form
	 */
	private function createDeleteForm(Article $article)
	{
		return $this->createFormBuilder()
			->setAction($this->generateUrl('webadmin_articles_delete', array('id' => $article->getId())))
			->setMethod('DELETE')
			->getForm()
		;
	}

	/**
	 * @param     $dql
	 * @param int $pageSize
	 * @param int $currentPage
	 *
	 * @return Paginator
	 */
	protected function _paginate($dql, $pageSize = 10, $currentPage = 1)
	{
		$paginator = new Paginator($dql);

		$paginator
			->getQuery()
			->setFirstResult($pageSize * ($currentPage - 1)) // set the offset
			->setMaxResults($pageSize); // set the limit

		$paginator->recordsOnPage = $pageSize;
		$paginator->recordCount = count($paginator);
		$paginator->current = $currentPage;
		$paginator->pageCount = ceil($paginator->recordCount / $pageSize);

		return $paginator;
	}


}
