<?php
namespace Dalten\WebAdminBundle\Processor;
use App\FrontBundle\Entity\Article;
use App\FrontBundle\Entity\ArticleHistory;
use \Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Class ArticleStorage
 *
 * Třída, která se stará o verzování článků.
 *
 * @package Dalten\WebAdminBundle\Processor
 */
class ArticleStorage
{
	/** @var \Doctrine\ORM\EntityManagerInterface  */
	protected $_em;

	/** @var  UserInterface */
	protected $_user;

	public function __construct($em)
	{
		$this->_em = $em;
	}

	public function setUser(UserInterface $user)
	{
		$this->_user = $user;
	}

	/**
	 * Zkontroluje, zda článek existuje. Pokud neexistuje, rovnou ho zapíše do DB.
	 *
	 * @param Article $article
	 */
	protected function _checkArticle(Article $article)
	{
		$em = $this->_em;

		$this->_checkSlugDuplicity($em, $article);

		if (!$article->getId()) {
			$article->setIsPublished(false);
			$article->setRevisionNumber(1);
			$em->persist($article);
			$em->flush();
			// protože je slug ve tvaru slug-ID, musíme ho po prvním uložení regenerovat, ehm ehm...
			$this->_checkSlugDuplicity($em, $article);
			$em->persist($article);
			$em->flush();
		}
	}

	protected function _saveRevision(Article $article, $isAutosave=false)
	{
		$em = $this->_em;
		$repo = $em->getRepository('App\FrontBundle\Entity\ArticleHistory');

		/** @var ArticleHistory|null $lastRevision */
		$lastRevision = $repo->findOneBy(array('idArticle'=>$article), array('revisionNumber'=>'DESC'));

		if (!$lastRevision) {
			$lastRevision = new ArticleHistory();
			$lastRevision->setRevisionNumber(0);
		}

		if ($lastRevision->hasChanges($article)) {
			$newRevision = new ArticleHistory();
			$newRevision->setIdArticle($article);

			$newRevision->setRevisionAuthor($this->_user->getUsername());
			$newRevision->setRevisionNumber($lastRevision->getRevisionNumber() + 1);
			$newRevision->setRevisionDate(new \DateTime('now'));
			$newRevision->setIsAutosave($isAutosave);

			$newRevision->setName($article->getName());
			$newRevision->setTeaser($article->getTeaser());
			$newRevision->setHtml($article->getHtml());

			$newRevision->setOldSlug($article->getOldSlug());

			$em->persist($newRevision);
			$em->flush();

			return $newRevision->getRevisionNumber();
		}
		return $lastRevision->getRevisionNumber();
	}

	/**
	 * Uloží změny článku, ale zatím je nepublikuje.
	 *
	 *
	 *
	 * @param Article $article
	 * @param bool    $isAutosave
	 */
	public function save(Article $article, $isAutosave=false)
	{
		$this->_checkArticle($article);
		return $this->_saveRevision($article, $isAutosave);
	}

	/**
	 * Uloží článek a publikuje ho.
	 *
	 * @param Article $article
	 */
	public function publish(Article $article)
	{
		$em = $this->_em;

		$this->_checkArticle($article);
		$revNum = $this->_saveRevision($article, false);

		$article->setRevisionNumber($revNum);
		$article->setIsPublished(true);

		$em->persist($article);
		$em->flush();
	}

	/**
	 * Skryje článek z webu.
	 *
	 * @param Article $article
	 */
	public function unpublish(Article $article)
	{

	}

	/**
	 * Získá článek $articleId ve verzi $revisionNum.
	 *
	 * @param $article
	 * @param $revisionNum
	 */
	public function getRevision(Article $article, $revisionNum)
	{
		$em = $this->_em;

		$revision = $em
			->getRepository('App\FrontBundle\Entity\ArticleHistory')
			->findOneBy(array('idArticle'=>$article, 'revisionNumber'=>$revisionNum));

		if (!$revision) {
			throw new NotFoundHttpException('Tato revize neexistuje.');
		}

		$article->setRevisionNumber($revision->getRevisionNumber());
		$article->setName($revision->getName());
		$article->setTeaser($revision->getTeaser());
		$article->setHtml($revision->getHtml());
		$article->setOldSlug($revision->getOldSlug());

		return $article;
	}

	/**
	 * Získá historii článku.
	 *
	 * @param Article $article
	 *
	 * @return array|\Dalten\WebAdminBundle\Entity\ArticleHistory[]
	 */
	public function getHistory(Article $article)
	{
		$repo = $this->_em->getRepository('App\FrontBundle\Entity\ArticleHistory');
		return $repo->findBy(array('idArticle'=>$article), array('revisionNumber'=>'DESC'));
	}

	/**
	 * Zajistí, aby nebyl duplikátní slug entity.
	 *
	 * @param object $em     Doctrine EntityManager.
	 * @param Article $entity Entita, u které chcem hlídat unikátnost slugu.
	 *
	 * @return object entita s opraveným slugem
	 */
	protected function _checkSlugDuplicity($em, $entity)
	{
		if (method_exists($entity, 'regenerateSlug')) {
			$entity->regenerateSlug();
		}
		$originalSlug = $entity->getSlug();

		if (!$entity->getId()) {
			$entity->setSlug($originalSlug . '-' . uniqid('#', true));
		} else {
			$entity->setSlug($originalSlug . '-' . $entity->getId());
		}
		return $entity;
	}
}