# BankID OpenId Connect Provider - JWK Authorization based on JWKs issued by your software

This is the most secure scenario of authorization. It's a bit more complex than the other two, so follow the next guide:

## Configuration

1. Generate the keys

In order to use this authorization stategy, you need to issue your own JWK.

The proposed way is to use `Jose\Component\KeyManagement\JWKFactory` which is wrapped in `BankId\OIDC\Tools\KeyGenerator`.

So, to obtain the keys, you need:

```php
$keyGenerator = new KeyGenerator(); //create an instance of the key generator

$privateKey = $keyGenerator->generate(); //generate the private key
$publicKey = $privateKey->toPublic(); //generate its' public print
```

Then simply store the serialized output of these entities in the way you prefer.
Here is the most basic example:

```php
file_put_contents('./private_key.json', json_encode($privateKey));
file_put_contents('./public_key.json', json_encode($publicKey));
```

2. Create BankId Provider:

```php
    $privateKey = JWK::createFromJson(file_get_contents('./private_key.json'));

    $provider = new BankIdProvider(
        settings: new Settings(
            bankIdBaseUri: 'https://oidc.sandbox.bankid.cz', //BankId instance base URI
            postLoginRedirectUri: 'http://localhost:3000', //your software URI + post-login redirect path
            postLogoutRedirectUri: 'http://localhost:3000/logout', //your software + post-logout redirect path
            clientId: 'd544ec7e-6391-40b0-afe6-601ede4b47fe', //your id acquired from BankId dashboard
            clientSecret: 'OgHok0p_bqesQGytk8YBb1PPjO6fL82ZUlAkO7fRfg-l6KhNQCt1t1h097de-CNj1a1JCJMViAM9N8MLcIml2Q', //your secret also acquired from BankId dashboard
            authStrategy: AuthStrategy::SignedWithOwnJWK, //choose exactly AuthStrategy::SignedWithOwnJWK here
            jwk: $privateKey, //make sure that Private Key instance is passed here. It's used to sign the outcoming packages
        ),
    );
```

3. Exposing the JWKs

Since the key is generated by RP (your software), BankId has to have an opportunity to verify the signature of your packages. But in order to follow the idea of asymmetric cryptography, we should not expose the private key directly. That's why it's private.

So we expose the public one, in the way OpenId Connect protocol advises.

Let's say, your software's JWK discovery endpoint is `https://acme.com/.well-known/jwks`. Remember, this URI has to be configured in BankId Dashboard in before.

Then, the response of this URI has to do the thing equal to this code:

```php
    $publicKey = JWK::createFromJson(file_get_contents('./public_key.json'));

    header('Content-type: application/json');
    echo json_encode([
        'keys' => [$publicKey],
    ]);
    exit();
```

For the local development, feel free to use something like [ngrok](https://ngrok.com) to accept the calls from the outside.

4. Exposing other endpoints

In the basic scenario, you don't need to expose any other endpoint except the one to accept the reverse redirect of the Client with "authorization code".
Once you receive this redirect, make sure to validate the state of the request and then to call `$provider->getClient($_GET['code'])` in order to get the configured BankId client.

5. Accepting the redirect

It's a good idea to verify that the `state` the client comes with is the same value as the one that has been generated at the time of the auth URI creation.

Once you perform all the checks, just pass the `code` get parameter to the provider in order to get the fully configured BankId client.

```php
$client = $provider->getClient($_GET['code']);
```


Please read [Quickstart](../QUICKSTART.md) chapter to know more about the flow of the authorization.